//----------------------------------------------------------
// Copyright (C) Microsoft Corporation. All rights reserved.
//----------------------------------------------------------
// TFS.UI.Controls.Grids.js
//
/*globals _comma_separated_list_of_variables_*/
/// <reference path="~/Scripts/DevTime.js" />
/// <reference path=~/Scripts/jquery-1.6.2-vsdoc.js" />
/// <reference path="~/Scripts/MicrosoftAjax.js"/>
/// <reference path="~/Scripts/TFS/TFS.js" />
/// <reference path="~/Scripts/TFS/TFS.Core.js" />
/// <reference path="~/Scripts/TFS/TFS.Diag.js" />
/// <reference path="~/Scripts/TFS/TFS.UI.js" />


TFS.module("TFS.UI.Controls.Grids", ["TFS.Resources.Presentation", "TFS.UI", "TFS.Core", "TFS.Core.Utils", "TFS.Diag", "TFS.UI.Controls", "TFS.UI.Controls.Data", "TFS.UI.Controls.Menus", "TFS.UI.Controls.Common"], function () {
    var log = TFS.Diag.log,
        Diag = TFS.Diag,
        verbose = TFS.Diag.LogVerbosity.Verbose,
        delegate = TFS.Core.delegate,
        domElem = TFS.UI.domElem,
        Controls = TFS.UI.Controls,
        MenuControls = TFS.UI.Controls.Menus,
        Resources = TFS.Resources.Presentation,
        CommonControls = TFS.UI.Controls.Common,
        TFSHOST = TFS.Host,
        Data = TFS.UI.Controls.Data,
        Core = TFS.Core,
        CoreUtils = TFS.Core.Utils;

    function Grid(options) {
        /// <summary>Creates new Grid Control</summary>
        /// <param name="options" type="object">The initialization options for the grid which have the following properties
        ///
        ///    "columns" is a required property containing the array of grid column descriptors that have the following structure:
        ///    {
        ///        index: The index for the
        ///        text:      <column header text, string, optional, default: "">,
        ///        tooltip:   <column header tooltip text, string, optional, default: "">,
        ///        width:     <width in pixels of the column, number, optional, default: 100>,
        ///        canSortBy: <true if the grid can be sorted by the column, boolean, optional, default: true>
        ///        canMove: <true if this column can be moved (has effect only if allowMoveColumns is set to true for the grid as well), boolean, optional, default: true>
        ///        getCellContents: <function that returns cell contents, function, optional, default: this._drawCell>
        ///            The function takes the same parameters as _drawCell and should return a jQuery object
        ///            that represents the cell's contents. The first element will be appended to the row.
        ///            If the function returns <c>null</c> or <c>undefined</c> nothing will be appended for that cell.
        ///        getHeaderCellContents: <function that returns column header cell contents, function, optional, default: this._drawHeaderCellValue>
        ///            The function takes the same parameters as _drawHeaderCellValue and should return a jQuery object
        ///            that represents the cell's contents. The first element will be appended to the header cell's contents.
        ///            If the function returns <c>null</c> or <c>undefined</c> nothing will be appended for that header cell.
        ///        getColumnValue: <function that returns the value for a cell contents, function, optional, default: this.getColumnValue;>
        ///            The return value of the function will be converted to a string an added as the cell contents.
        ///            The function should return the text for that cell's tooltip if the isForTooltip parameter is set to true.
        ///        getCellCSSClass: <function that returns the value for a cell style, function, optional>
        ///            The return value of the function will be added to the style of the cell element.
        ///    }
        ///    "enabledEvents" is an optional property containing an object with properties for each of the enabled events.
        ///    {
        ///        Grid.EVENT_ROW_UPDATED: true
        ///    }
        ///
        /// </param>

        this.baseConstructor.call(this, $.extend({
            autoSort: true,
            payloadSize: Grid.PAYLOAD_SIZE, /* Default viewport size, keep in sync with WorkItemManager.PAGE_SIZE */
            extendViewportBy: 3,
            gutter: {
                contextMenu: false,
                icon: false,
                checkbox: false
            },
            header: true,
            allowMoveColumns: false,      // Controls whether columns can be moved in the grid
            draggable: null,             // JQuery UI Dragable options for grid rows.
            droppable: null,              // JQuery UI Droppable options for grid rows.
            asyncInit: true,
            sharedMeasurements: true,
            coreCssClass: "grid",
            allowSortOnMultiColumns: true
        }, options));

        this._rows = {};
    }

    Grid.extend({
        _typeName: "tfs.grid",
        MAX_COPY_SIZE: 1000,
        PAYLOAD_SIZE: 200,
        EVENT_ROW_UPDATED: "rowupdated",                      // Name for the row updated event.
        EVENT_SELECTED_INDEX_CHANGED: "selectedIndexChanged"  // Name for the selected index changed event
    });

    Grid.inherit(Controls.BaseControl, (function () {
        var measurements;

        function expand(states) {
            var result = [];

            if (states.length > 0) {
                var stack = [];
                var currState = { level: 1, origCount: states.length, remainingCount: states.length };
                stack.push(currState);

                var i = 0;
                while (i < states.length) {
                    result.push(currState.level);
                    currState.remainingCount--;
                    while (currState.remainingCount === 0) {
                        stack.pop();
                        if (stack.length == 0) {
                            if (i === (states.length - 1)) {
                                break;
                            } else {
                                throw new Error("invalid descendant counts, nesting not possible");
                            }
                        }
                        var newState = stack[stack.length - 1];
                        newState.remainingCount -= currState.origCount;
                        currState = newState;
                        if (currState.remainingCount < 0) {
                            throw new Error("invalid descendant counts, cannot convert to indentation levels");
                        }
                    }

                    var nextCount = Math.abs(states[i]);

                    if (nextCount > 0) {
                        var nextItem = { level: result[result.length - 1] + 1, origCount: nextCount, remainingCount: nextCount };
                        stack.push(nextItem);
                        currState = nextItem;
                    }
                    i++;
                }

                if (stack.length > 0) {
                    throw new Error("invalid descendant counts, more input expected");
                }
            }

            return result;
        }

        // To ensure sorting only siblings, we need to convert the flat list to tree.
        // This class is used to represent a node in the tree.
        function TreeNode(data, expandState) {
            this.data = data;
            this.expandState = expandState;
            this.children = [];
        }

        function addItemsToTree(
            dataSource,    // TFS Grid's _dataSource
            expandStates,  // TFS Grid's _expandStates
            start,         // start index of the sub array
            count,         // length of the sub array
            parentNode     // parent node of all the items in the sub array
            ) {
            var end = Math.min(start + count, dataSource.length);
            for (var i = start; i < end; i++) {
                var node = new TreeNode(dataSource[i], expandStates[i]);
                var childrenCount = Math.abs(node.expandState);
                if (childrenCount > 0) {
                    addItemsToTree(dataSource, expandStates, i + 1, childrenCount, node);
                    i += childrenCount;
                }

                parentNode.children.push(node);
            }
        }

        return {
            _dataSource: [],
            _expandStates: null,
            _indentLevels: null,
            _columns: [],
            _sortOrder: [],
            _visibleRange: [],
            _count: 0,
            _expandedCount: 0,
            _selectedIndex: -1,

            _indentIndex: 0,
            /* selection */
            _selectionStart: -1,
            _selectionCount: 0,
            _selectedRows: null,

            _rowHeight: 20,
            _cellOffset: 0,
            _gutterWidth: 30,
            _contentSize: null,
            _rows: {},
            _focus: null,
            _scroller: null,
            _canvas: null,
            _canvasHeight: 300,
            _canvasWidth: 300,
            _contentSpacer: null,
            _header: null,
            _headerCanvas: null,
            _gutter: null,
            _gutterHeader: null,
            _popupMenu: null,

            _resetScroll: false,
            _ignoreScroll: false,
            _scrollTop: 0,
            _scrollLeft: 0,
            _ddRowAcceptStatus: null,
            _ddRowOverStatus: null,
            _droppable: null,
            _draggable: null,
            _draggingRowInfo: null,
            _cancelable: null,

            _active: false,
            _activeAriaId: null,
            _getChildDataCallback: null,
            _rowClickCallback: null,

            getSelectionCount: function () { return this._selectionCount; },

            _createElement: function () {
                this._base();

                this._buildDom();
            },

            _enhance: function (element) {
                this._base(element);

                this._buildDom();
            },

            _buildDom: function () {
                var fragment = document.createDocumentFragment(), gutterOptions, gutterVisible;

                gutterOptions = this._options.gutter;

                gutterVisible = gutterOptions && (gutterOptions.contextMenu || gutterOptions.icon || gutterOptions.checkbox);

                this._focus = $(domElem("div", "grid-focus")).attr("tabIndex", 0).attr("role", "grid");
                fragment.appendChild(this._focus[0]);

                this._canvas = $(domElem("div", "grid-canvas"));
                this._contentSpacer = $(domElem("div", "grid-content-spacer"));
                this._canvas.append(this._contentSpacer);

                if (this._options.header) {
                    this._element.addClass("has-header");
                    this._header = $(domElem("div", "grid-header"));

                    this._headerCanvas = $(domElem("div", "grid-header-canvas"));
                    this._header.append(this._headerCanvas);

                    fragment.appendChild(this._header[0]);
                }

                if (gutterVisible) {
                    this._element.addClass("has-gutter");
                    this._gutter = $(domElem("div", "grid-gutter"));
                    this._canvas.append(this._gutter);

                    if (this._header) {
                        this._gutterHeader = $(domElem("div", "grid-gutter-header"));
                        this._header.append(this._gutterHeader);
                    }
                }

                fragment.appendChild(this._canvas[0]);
                this._element.append(fragment);
            },

            initialize: function () {
                this._base();

                this._contentSize = { width: 300, height: 400 };
                this._takeMeasurements();
                this._getChildDataCallback = this._options.childDataCallback || null;
                this._rowClickCallback = this._options.rowClickCallback || null;

                //attach events later. it takes very long time to execute this function. It prevents work item form to display late
                //if there are multiple links control.
                if (this._options.asyncInit) {
                    Core.delay(this, 10, function () {
                        this._attachEvents();
                    });
                }
                else {
                    this._attachEvents();
                }

                this.initializeDataSource();

                TFS.Diag.logTracePoint('Grid.initialize.complete');
            },

            _attachEvents: function () {
                var that = this;

                this._bind(window, "resize", delegate(this, this._onContainerResize));

                this._bind("mousedown", delegate(this, this._onContainerMouseDown));

                this._bind(this._focus, "keydown", delegate(this, this._onKeyDown));
                this._bind(this._focus, "keyup", delegate(this, this._onKeyUp));
                this._bind(this._focus, "focus", delegate(this, this._onFocus));
                this._bind(this._focus, "blur", delegate(this, this._onBlur));

                if (this._gutter && (this._options.gutter && this._options.gutter.contextMenu)) {
                    this._bind(this._focus, "contextmenu", delegate(this, this._onContextMenu));
                    this._bind("contextmenu", delegate(this, this._onContextMenu));
                }

                this._bind(this._canvas, "click", delegate(this, this._onRowClick));
                this._bind(this._canvas, "dblclick", delegate(this, this._onRowDoubleClick));
                this._bind(this._canvas, "mousedown", delegate(this, this._onRowMouseDown));
                this._bind(this._canvas, "scroll", delegate(this, this._onCanvasScroll));
                this._bind(this._canvas, "selectstart", function () { return false; });

                if (this._header) {
                    // Binding the necessary events for column move, resize and sort
                    this._bind(this._header, "mousedown", delegate(this, this._onHeaderMouseDown));
                    this._bind(this._header, "mouseup", delegate(this, this._onHeaderMouseUp));
                    this._bind(this._header, "click", delegate(this, this._onHeaderClick));
                    this._bind(this._header, "dblclick", delegate(this, this._onHeaderDblClick));
                }

                if (this._gutter) {
                    this._bind(this._gutter, "click", delegate(this, this._onGutterClick));

                    this._gutter.hover(function (e) {
                        $(e.target)
                                    .closest(".grid-gutter-row")
                                    .addClass("grid-gutter-row-hover");
                    }, function (e) {
                        $(e.target)
                                    .closest(".grid-gutter-row")
                                    .removeClass("grid-gutter-row-hover");
                    });
                }

                this._setupDragDrop();

                this._element.closest(".ui-tabs").bind("tabsshow", function (e, ui) {
                    if (that._element.is(":visible")) {
                        that.layout();
                    }
                });

                TFS.Diag.logTracePoint('Grid._attachEvents.complete');
            },

            _setupDragDrop: function () {
                var grid = this, draggableOptions = this._options.draggable, draggableDefaults, draggable, draggingRowInfo;

                if (draggableOptions) {
                    draggableDefaults = {
                        axis: "y",
                        helperClass: "grid-row-dragging",
                        zIndex: 1000,
                        appendTo: this._canvas,
                        handle: ".grid-row",
                        opacity: 0.7,
                        scope: Grid.getTypeName(),
                        cursorAt: { top: this._rowHeight / 2 },
                        dropBehaviour: true //help jquery ui here not to invoke droppables
                    };

                    if (typeof draggableOptions !== "object") {
                        draggableOptions = {};
                    }

                    this._draggable = $.extend({}, draggableDefaults, draggableOptions);

                    draggableOptions = $.extend(draggableDefaults,
                                                draggableOptions,
                    {
                        helper: function (e) {
                            var target = $(e.target), row, rowInfo;

                            row = target.closest(".grid-row");

                            if (row.length) {
                                rowInfo = row.data("grid-row-info");
                                draggingRowInfo = { dataIndex: +rowInfo.dataIndex, rowIndex: +rowInfo.rowIndex };

                                return grid._rowDragCreateHelper(draggingRowInfo, e, draggable);
                            }
                            else {
                                //it seems user started dragging in an empty grid
                                rowInfo = null;
                                draggingRowInfo = null;
                                return $("<div />");
                            }
                        },
                        drag: function (e, ui) {
                            ui.draggable = draggable;
                            ui.draggingRowInfo = draggingRowInfo;
                            return grid._rowDrag(draggingRowInfo, e, ui);
                        },
                        start: function (e, ui) {
                            if (draggingRowInfo) {
                                ui.draggable = draggable;
                                ui.draggingRowInfo = draggingRowInfo;

                                // When the "ui" instance is passed into the droppable handlers it is not the same
                                // instance as we have here, so any additional data we have set on it here will not
                                // be available to these handlers.  To ensure droppable handlers have access to the
                                // draging row info, save it off.
                                grid._draggingRowInfo = draggingRowInfo;
                                return grid._rowDragStart(draggingRowInfo, e, ui);
                            }
                            else {
                                return false; //stop dragging
                            }
                        },
                        stop: function (e, ui) {
                            ui.draggable = draggable;
                            ui.draggingRowInfo = draggingRowInfo;

                            // Clear the dragging row info since we are done dragging.
                            grid._draggingRowInfo = null;
                            return grid._rowDragStop(draggingRowInfo, e, ui);
                        }
                    });

                    this._droppable = {
                        tolerance: "intersect",
                        hoverClass: "grid-row-drop-active"
                    };

                    if (typeof this._options.droppable === "object") {
                        $.extend(this._droppable, this._options.droppable);

                        if (this._droppable.accept) {
                            if (!$.isFunction(this._droppable.accept)) {
                                this._droppable.acceptSelector = this._droppable.accept;
                            }
                        }
                    }

                    this._canvas.draggable(draggableOptions);
                    draggable = this._canvas.data("draggable");

                    this._resetRowAcceptStatus();
                    this._resetRowOverStatus();
                }
            },

            _mergeExpandStates: function (parentIndex, oldExpandStates, newExpandStates) {

                var netIncreaseInExpandStates = newExpandStates.length - 1;  // minus 1 because placeholder will be gone
                oldExpandStates.splice(parentIndex + 1, 1);  // remove placeholder

                for (var i = 0; i <= netIncreaseInExpandStates; i++) {
                    oldExpandStates.splice(parentIndex + i + 1, 0, newExpandStates[i]);
                }

                var countSinceLastParent = 0;
                for (var i = parentIndex; i >= 0; i--) {
                    var origValue = oldExpandStates[i];

                    if (Math.abs(origValue) > countSinceLastParent) {  // if this is true we've encountered a parent state
                        if (origValue < 0) {
                            oldExpandStates[i] = origValue - netIncreaseInExpandStates;
                        } else {
                            oldExpandStates[i] = origValue + netIncreaseInExpandStates;
                        }
                        countSinceLastParent = 0;
                    } else {
                        countSinceLastParent++;
                    }
                }

            },

            getDraggingRowInfo: function () {
                /// <summary>Gets the row information for the item currently being dragged.</summary>
                /// <returns type="object" />
                return this._draggingRowInfo;
            },

            _getDragOverRows: function () {
                /// <summary>Get the rows that currently have a draggable item "over" them</summary>
                return this._ddRowOverStatus;
            },

            _resetRowOverStatus: function () {
                /// <summary>Clear the record of which rows the draggable objects are "over"</summary>
                this._ddRowOverStatus = {};
            },

            _resetRowAcceptStatus: function () {
                /// <summary>Clear the cached row acceptance map</summary>
                this._ddRowAcceptStatus = {};
            },

            _rowDropAccept: function (droppingRowInfo, draggingRowInfo, e, ui) {
                var rowInfo = this._rows[droppingRowInfo.dataIndex], accept = this._droppable.accept, acceptSelector;

                if (rowInfo) {
                    if (accept) {
                        return accept.call(this, e, $.extend({}, ui, { droppingRowInfo: rowInfo }));
                    }
                    acceptSelector = this._droppable.acceptSelector;
                    if (acceptSelector) {
                        return rowInfo.row.is(acceptSelector);
                    }

                    return true;
                }
                else {
                    return false;
                }
            },

            _rowDropActivate: function (droppingRowInfo, draggingRowInfo, e, ui) {
                var rowInfo = this._rows[droppingRowInfo.dataIndex], activate = this._droppable.activate;

                if (rowInfo) {
                    if (this._droppable.activeClass) {
                        rowInfo.row.addClass(this._droppable.activeClass);
                    }

                    if (activate) {
                        return activate.call(this, e, $.extend({}, ui, { droppingRowInfo: rowInfo }));
                    }
                }
            },

            _rowDropDeactivate: function (droppingRowInfo, draggingRowInfo, e, ui) {
                var rowInfo = this._rows[droppingRowInfo.dataIndex], deactivate = this._droppable.deactivate;

                if (rowInfo) {
                    if (this._droppable.activeClass) {
                        rowInfo.row.removeClass(this._droppable.activeClass);
                    }

                    if (deactivate) {
                        return deactivate.call(this, e, $.extend({}, ui, { droppingRowInfo: rowInfo }));
                    }
                }
            },

            _rowDropTryActivate: function (droppingRowInfo, draggingRowInfo, e, ui) {
                var accepted = this._ddRowAcceptStatus[droppingRowInfo.dataIndex];
                if (typeof accepted === "undefined") {
                    this._ddRowAcceptStatus[droppingRowInfo.dataIndex] = accepted = this._rowDropAccept(droppingRowInfo, draggingRowInfo, e, ui);
                    if (accepted) {
                        this._rowDropActivate(droppingRowInfo, draggingRowInfo, e, ui);
                    }
                }

                return accepted;
            },

            _rowDropOver: function (droppingRowInfo, draggingRowInfo, e, ui) {
                var rowInfo = this._rows[droppingRowInfo.dataIndex], droppable = this._droppable, over = this._droppable.over;
                if (rowInfo) {
                    ui.draggable.dropped = true;
                    if (droppable.hoverClass) {
                        rowInfo.row.addClass(droppable.hoverClass);
                    }

                    if (over) {
                        over.call(this, e, $.extend({}, ui, { droppingRowInfo: rowInfo }));
                    }
                }
            },

            _rowDropOut: function (droppingRowInfo, draggingRowInfo, e, ui) {
                var rowInfo = this._rows[droppingRowInfo.dataIndex], droppable = this._droppable, out = this._droppable.out;
                if (rowInfo) {
                    ui.draggable.dropped = false;
                    if (droppable.hoverClass) {
                        rowInfo.row.removeClass(droppable.hoverClass);
                    }

                    if (out) {
                        out.call(this, e, $.extend({}, ui, { droppingRowInfo: rowInfo }));
                    }
                }
            },

            _rowDrop: function (droppingRowInfo, draggingRowInfo, e, ui) {
                var rowInfo = this._rows[droppingRowInfo.dataIndex], droppable = this._droppable;
                if (rowInfo) {
                    if (droppable.hoverClass) {
                        rowInfo.row.removeClass(droppable.hoverClass);
                    }

                    if (droppable.activeClass) {
                        rowInfo.row.removeClass(droppable.activeClass);
                    }

                    if (droppable.drop) {
                        droppable.drop.call(this, e, $.extend({}, ui, { droppingRowInfo: rowInfo }));
                    }
                    // Removed auto-row selection. The responsibility for selecting the updated rows
                    // lies with the drop handler.
                }
            },

            _rowDragCreateHelper: function (draggingRowInfo, e, ui) {
                var helper;

                if ($.isFunction(this._draggable.helper)) {
                    return this._draggable.helper.call(this, e, $.extend({}, ui, { draggingRowInfo: draggingRowInfo }));
                }

                helper = this._rows[draggingRowInfo.dataIndex].row.clone();
                helper.removeClass("grid-row-selected grid-row-selected-blur grid-row-current");
                helper.addClass(this._draggable.helperClass);
                helper.css("position", "absolute");

                return helper;
            },

            _rowDrag: function (draggingRowInfo, e, ui) {
                var that = this, overStatus = this._getDragOverRows(), drag = this._draggable.drag;

                if (drag && drag.call(this, e, ui) === false) {
                    return false;
                }

                $.each(this._rows, function (dataIndex, rowInfo) {
                    var droppingRowInfo;
                    if (+dataIndex !== +draggingRowInfo.dataIndex) {
                        droppingRowInfo = { dataIndex: +dataIndex, rowIndex: +rowInfo.rowIndex };

                        if (that._rowDropTryActivate(droppingRowInfo, draggingRowInfo, e, ui)) {
                            if (that._rowIntersect(ui.draggable, droppingRowInfo)) {
                                if (!overStatus[dataIndex]) {
                                    overStatus[dataIndex] = true;
                                    that._rowDropOver(droppingRowInfo, draggingRowInfo, e, ui);
                                }
                            }
                            else {
                                if (overStatus[dataIndex]) {
                                    delete overStatus[dataIndex];
                                    that._rowDropOut(droppingRowInfo, draggingRowInfo, e, ui);
                                }
                            }
                        }
                    }
                });
            },

            _rowDragStart: function (draggingRowInfo, e, ui) {
                var that = this, start = this._draggable.start;

                if (start && start.call(this, e, ui) === false) {
                    return false;
                }

                $.each(this._rows, function (dataIndex, rowInfo) {
                    if (+dataIndex !== draggingRowInfo.dataIndex) {
                        that._rowDropTryActivate({ dataIndex: +dataIndex, rowIndex: rowInfo.rowIndex }, draggingRowInfo, e, ui);
                    }
                });
            },

            _rowDragStop: function (draggingRowInfo, e, ui) {
                var that = this, stop = this._draggable.stop;

                if (stop && stop.call(this, e, ui) === false) {
                    return false;
                }

                $.each(this._rows, function (dataIndex, rowInfo) {
                    var droppingRowInfo,
                        result;

                    if (+dataIndex !== +draggingRowInfo.dataIndex) {
                        droppingRowInfo = { dataIndex: Number(dataIndex), rowIndex: rowInfo.rowIndex };

                        if (that._rowDropTryActivate(droppingRowInfo, draggingRowInfo, e, ui)) {
                            if (that._rowIntersect(ui.draggable, droppingRowInfo)) {
                                // Now that we have found a row that intersects, do not check subsequent rows.
                                result = false;

                                that._rowDrop(droppingRowInfo, draggingRowInfo, e, ui);
                            }

                            that._rowDropDeactivate(droppingRowInfo, draggingRowInfo, e, ui);
                        }
                    }

                    return result;
                });

                this._resetRowAcceptStatus();
                this._resetRowOverStatus();
            },

            _rowIntersect: function (draggable, targetRowInfo) {
                var rowInfo = this._rows[targetRowInfo.dataIndex], row;

                if (rowInfo) {
                    row = rowInfo.row;
                    return $.ui.intersect(draggable, {
                        offset: row.offset(),
                        proportions: {
                            width: row[0].offsetWidth,
                            height: row[0].offsetHeight
                        }
                    },
                    this._droppable.tolerance);
                }

                return false;
            },

            _takeMeasurements: function () {
                var measurementContainer, row, cell, textUnit, gutter, currentMeasurements, cssClass;

                if (this._options.sharedMeasurements) {
                    currentMeasurements = measurements;
                }

                if (!currentMeasurements) {
                    currentMeasurements = {};
                    cssClass = this._options.coreCssClass;

                    if (this._options.cssClass) {
                        cssClass += " " + this._options.cssClass;
                    }

                    measurementContainer = $(domElem("div", cssClass)).appendTo(document.body)
                    .css("position", "absolute")
                    .css("left", -5000).css("top", -5000)
                    .width(1000).height(500);

                    row = $(domElem("div", "grid-row grid-row-normal")).appendTo(measurementContainer);
                    cell = $(domElem("div", "grid-cell")).width(100).appendTo(row).text("1");

                    currentMeasurements.rowHeight = row.outerHeight();
                    currentMeasurements.cellOffset = cell.outerWidth() - 100;

                    textUnit = $(domElem("div")).appendTo(cell)
                                                .css("overflow", "hidden")
                                                .width("1em").height("1ex");
                    currentMeasurements.unitEx = textUnit.outerHeight();

                    gutter = $(domElem("div", "grid-gutter"));
                    gutter.append(domElem("div", "grid-gutter-row grid-gutter-row-selected"));
                    gutter.appendTo(measurementContainer);

                    currentMeasurements.gutterWidth = gutter[0].clientWidth;
                    measurementContainer.remove();

                    if (this._options.sharedMeasurements) {
                        measurements = currentMeasurements;
                    }
                }

                this._unitEx = currentMeasurements.unitEx;

                this._rowHeight = currentMeasurements.rowHeight;
                this._cellOffset = currentMeasurements.cellOffset;

                if (this._gutter) {
                    this._gutterWidth = currentMeasurements.gutterWidth;
                }
                else {
                    this._gutterWidth = 0;
                }
            },

            initializeDataSource: function () {
                var canvas;
                if (this._resetScroll) {
                    this._ignoreScroll = true;
                    try {
                        //jQuery ScrollLeft and ScrollTop methods are slow use direct dom
                        canvas = this._canvas[0];
                        canvas.scrollTop = 0;
                        canvas.scrollLeft = 0;
                        this._scrollLeft = 0;
                        this._scrollTop = 0;
                        this._resetScroll = false;
                    }
                    finally {
                        this._ignoreScroll = false;
                    }
                }

                this.setDataSource(this._options.source, this._options.expandStates, this._options.columns, this._options.sortOrder);

                if (this._expandedCount > 0) {
                    if (this._options.keepSelection && this._selectedIndex >= 0) {
                        this._selectRow(Math.min(this._selectedIndex, this._expandedCount - 1));
                    }
                    else {
                        this._selectRow(this._options.initialSelection !== false ? 0 : -1);
                    }

                }
                else {
                    this.setSelectedRowIndex(-1);
                }
            },

            setDataSource: function (source, expandStates, columns, sortOrder, selectedIndex) {
                var i, l, count, column, that = this;

                this._dataSource = source || [];
                this._count = count = this._dataSource.length;

                if (expandStates) {
                    this._expandStates = expandStates;
                    this._indentLevels = expand(expandStates);
                } else {
                    this._indentLevels = null;
                    this._expandStates = null;
                }

                this._expandedCount = count;
                this._updateRanges();

                this._columns = [];

                if (columns) {
                    //columns are optional.

                    for (i = 0, l = columns.length; i < l; i++) {
                        column = columns[i];

                        // Column value default settings
                        column.index = typeof (column.index) !== "undefined" ? column.index : i;
                        column.canSortBy = column.canSortBy !== false;
                        column.canMove = column.canMove !== false;
                        column.width = typeof (column.width) !== "undefined" ? column.width : 100;

                        // Column drawing default implementations
                        column.getCellContents = column.getCellContents || this._drawCell;
                        column.getHeaderCellContents = column.getHeaderCellContents || this._drawHeaderCellValue;
                        column.getColumnValue = column.getColumnValue || this.getColumnValue;
                        this._columns.push(column);
                    }
                }

                this._sortOrder = [];

                if (sortOrder) {
                    for (i = 0, l = sortOrder.length; i < l; i++) {
                        column = sortOrder[i];
                        if (column.order !== "desc") {
                            column.order = "asc";
                        }

                        this._sortOrder.push(column);
                    }
                }

                this._clearSelection();

                this._determineIndentIndex();

                function layout() {
                    that.layout();
                    that._ensureSelectedIndex(selectedIndex);
                }

                if (this._options.asyncInit) {
                    Core.delay(this, 0, layout);
                }
                else {
                    layout();
                }
            },


            _adjustForDynamicData: function (newRows, newExpandStates, parentIndex) {
                /// <summary>
                ///     Modifies the datasource and expandStates to either insert new data or remove 
                ///     placeholder if no new rows were returned
                /// </summary>

                this._dataSource.splice(parentIndex + 1, 1);

                for (var i = 0; i < newRows.length; i++) {
                    this._dataSource.splice(parentIndex + i + 1, 0, newRows[i]);
                }

                this._mergeExpandStates(parentIndex, this._expandStates, newExpandStates);

                count = this._dataSource.length;
                this._count = count;

                if (this._expandStates) {
                    this._indentLevels = expand(this._expandStates);
                } else {
                    this._indentLevels = null;
                }

                this._expandedCount = count;
                this._updateRanges();

                this._clearSelection();

                this._determineIndentIndex();

                this.layout();
            },

            _ensureSelectedIndex: function (index) {
                /// <summary>
                ///     Ensures that the selected index is correctly set. That is, it will be a noop if the index doesnt change
                ///     and will handle indexes that are out of bounds.
                /// </summary>
                /// <param name="index" type="Number">OPTIONAL: The index to select</param>

                var oldSelectedIndex = this._selectedIndex;

                if (typeof index === "number") { // If an index was passed then set it as the selected
                    this._selectedIndex = index;
                }

                if (this._selectedIndex >= 0) { // If there is a selected index then do something
                    if (this._count <= this._selectedIndex) { // If the selected index is greater than the number of visible rows then set it to the last row
                        this._selectedIndex = this._count - 1;
                    }

                    if (this._selectedIndex !== oldSelectedIndex) { // If the selected index changed then trigger selection changed
                        this._addSelection(this._selectedIndex);
                    }
                }
            },

            getRowInfo: function (dataIndex) {
                /// <summary>Gets the information about a row associated with the given data index</summary>
                /// <param name="dataIndex" type="number">The data index for the record to retrieve</param>
                /// <returns>A rowInfo object containing the rowIndex, dataIndex and a jQuery wrapper for the actual row</returns>
                Diag.assertParamIsNumber(dataIndex, "dataIndex");
                return this._rows[dataIndex];
            },

            getRowData: function (dataIndex) {
                /// <summary>Gets the data being used to display the row at the provided data index.</summary>
                /// <param name="dataIndex" type="number">The data index for the record to retrieve.</param>
                /// <returns type="object">Data being used to display the row.<returns>
                Diag.assertParamIsNumber(dataIndex, "dataIndex");

                return this._dataSource[dataIndex];
            },

            getColumns: function () {
                /// <summary>Gets the columns currently being displayed in the grid.</summary>

                return this._columns || [];
            },

            getSortOrder: function () {
                /// <summary>Gets the current sort order being used in the grid.</summary>

                return this._sortOrder || [];
            },

            setColumnOptions: function (columnName, options) {
                /// <summary>Set column hidden flag</summary>
                /// <param name="columnName" type="String">column Name</param>
                /// <param name="options" type="boolean">options</param>

                Diag.assertParamIsString(columnName, "columnName");
                Diag.assertParamIsObject(options, "options");

                var i, l,
                    columns = this._columns;

                for (i = 0, l = columns.length; i < l; i++) {
                    if (columns[i].name === columnName) {
                        columns[i] = $.extend(columns[i], options);
                        break;
                    }
                }
            },

            _determineIndentIndex: function () {
                var _columns = this._columns, i, l;
                for (i = 0, l = _columns.length; i < l; i++) {
                    if (_columns[i].indent) {
                        this._indentIndex = i;
                        return;
                    }
                }
                this._indentIndex = 0;
            },

            _getDataIndex: function (visibleIndex) {
                var i, l, lastIndex = -1, ranges = this._visibleRange, range;

                if (visibleIndex < 0) {
                    return -1;
                }

                for (i = 0, l = ranges.length; i < l; i++) {
                    range = ranges[i];
                    lastIndex += range[1] - range[0] + 1;

                    if (visibleIndex <= lastIndex) {
                        return range[1] - lastIndex + visibleIndex;
                    }
                }

                return visibleIndex;
            },

            _getRowIndex: function (dataIndex) {
                var i, l, result = 0, ranges = this._visibleRange, range;

                for (i = 0, l = ranges.length; i < l; i++) {
                    range = ranges[i];
                    if (dataIndex >= range[0]) {
                        if (dataIndex <= range[1]) {
                            return result + dataIndex - range[0];
                        }
                    }
                    else {
                        break;
                    }

                    result += range[1] - range[0] + 1;
                }

                return -Math.max(0, result - 1);
            },

            _updateRanges: function () {
                var i = 0, first = 0, l = this._count, newRanges = [], count = 0, state;

                if (this._expandStates) {
                    while (i < l) {
                        state = this._expandStates[i];

                        if (state < 0) {
                            newRanges[newRanges.length] = [first, i];
                            count += (i - first) + 1;
                            i += 1 - state;
                            first = i;
                        }
                        else {
                            i++;
                        }
                    }

                    if (first < l) {
                        newRanges[newRanges.length] = [first, l - 1];
                        count += (l - first);
                    }
                }
                else {
                    count = l;
                    newRanges[newRanges.length] = [0, count];
                }

                this._expandedCount = count;
                this._visibleRange = newRanges;
            },

            expandNode: function (dataIndex) {
                var state, row;

                if (this._dataSource[dataIndex + 1].isPlaceholder) {
                    var dynamicData = this._getChildDataCallback(this._dataSource[dataIndex]);

                    if (dynamicData !== null) {
                        this._adjustForDynamicData(dynamicData.itemsWithPlaceholders, dynamicData.expandStates, dataIndex);
                    }
                }

                if (this._expandStates) {
                    state = this._expandStates[dataIndex];

                    if (state < 0) {
                        this._expandStates[dataIndex] = -state;
                        this._updateRanges();
                        row = this._rows[dataIndex];
                        if (row) {
                            row.dirty = true;
                        }
                    }
                }
            },

            collapseNode: function (dataIndex) {
                var state, row;

                if (this._expandStates) {
                    state = this._expandStates[dataIndex];

                    if (state > 0) {
                        this._expandStates[dataIndex] = -state;
                        this._updateRanges();

                        row = this._rows[dataIndex];
                        if (row) {
                            row.dirty = true;
                        }
                    }
                }
            },

            expandAllNodes: function () {
                var i = 0, l = this._count, states = this._expandStates, state, result = false, row, rows = this._rows;

                if (states) {
                    while (i < l) {
                        state = states[i];
                        if (state < 0) {
                            states[i] = -state;
                            result = true;

                            row = rows[i];
                            if (row) {
                                row.dirty = true;
                            }
                        }

                        i++;
                    }

                    if (result) {
                        this._updateRanges();
                    }
                }

                return result;
            },

            collapseAllNodes: function () {
                var i = 0, l = this._count, states = this._expandStates, state, result = false, row, rows = this._rows;

                if (states) {
                    while (i < l) {
                        state = states[i];
                        if (state > 0) {
                            states[i] = -state;
                            result = true;

                            row = rows[i];
                            if (row) {
                                row.dirty = true;
                            }
                        }

                        i++;
                    }

                    if (result) {
                        this._updateRanges();
                    }
                }

                return result;
            },

            expandAll: function () {
                this._updateExpansionStateAndRedraw.call(this, this.expandAllNodes);
            },

            collapseAll: function () {
                this._updateExpansionStateAndRedraw.call(this, this.collapseAllNodes);
            },

            _updateExpansionStateAndRedraw: function (action) {
                Diag.assertParamIsFunction(action, "action");
                var dataIndex,
                    oldSelectedIndex = this._selectedIndex;

                if (oldSelectedIndex >= 0) {
                    dataIndex = this._getDataIndex(oldSelectedIndex);
                }

                action.call(this);  // expand or collapse all

                if (oldSelectedIndex >= 0) {
                    this._clearSelection();
                    this._addSelection(Math.abs(this._getRowIndex(dataIndex)));
                }

                this._layoutContentSpacer();
                this._redraw();
            },

            tryToggle: function (expand, shiftKey) {
                var row, dataIndex, state;

                if (!this._expandStates || this._selectedIndex < 0 || this._expandedCount <= 0) {
                    return false;
                }

                dataIndex = this._getDataIndex(this._selectedIndex);

                row = this._rows[dataIndex];

                if (!row) {
                    return false;
                }

                // No support for expand-all as this doesn't work correctly when the tree adds children only on expand
                ////if (shiftKey) {
                ////    if (expand) {
                ////        if (!this.expandAllNodes()) {
                ////            return false;
                ////        }
                ////    }
                ////    else {
                ////        if (!this.collapseAllNodes()) {
                ////            return false;
                ////        }
                ////    }

                ////    this._clearSelection();
                ////    this._addSelection(Math.abs(this._getRowIndex(dataIndex)));
                ////    this._layoutContentSpacer();
                ////    this._redraw();
                ////    return true;
                ////}
                ////else 
                {
                    state = this._expandStates[dataIndex];
                    if (state !== 0) {
                        if (expand) {
                            if (state < 0) {
                                this.expandNode(dataIndex);
                            }
                            else {
                                return false;
                            }
                        }
                        else {
                            if (state > 0) {
                                this.collapseNode(dataIndex);
                            }
                            else {
                                return false;
                            }
                        }

                        this._clearSelection();
                        this._addSelection(this._getRowIndex(dataIndex));
                        this._layoutContentSpacer();
                        this._redraw();
                        return true;
                    }
                }

                return false;
            },

            _getVisibleRowIndices: function () {
                var top = this._scrollTop,
                    bottom = top + this._canvasHeight,
                    count = this._expandedCount - 1,
                    rh = this._rowHeight;

                return {
                    first: Math.min(count, Math.max(0, Math.ceil(top / rh))),
                    last: Math.min(count, Math.floor(bottom / rh) - 1)
                };
            },

            _getRowIntoView: function (rowIndex, force) {
                var visibleIndices, firstIndex, lastIndex, count;

                if (force) {
                    // update view port will be called when scrolling happen
                    this._canvas[0].scrollTop = Math.max(0, Math.min(rowIndex || 0, this._expandedCount - 1)) * this._rowHeight;

                    return true;
                }

                visibleIndices = this._getVisibleRowIndices();
                firstIndex = visibleIndices.first;
                lastIndex = visibleIndices.last;

                count = lastIndex - firstIndex;


                if (rowIndex < firstIndex || rowIndex > lastIndex) {
                    if (this._selectedIndex > firstIndex) {
                        //set last visible
                        firstIndex = Math.max(rowIndex - count, 0);
                    }
                    else {//set first visible
                        firstIndex = Math.max(0, Math.min(rowIndex + count, this._expandedCount - 1) - count);
                    }

                    // update view port will be called when scrolling happen
                    this._canvas[0].scrollTop = firstIndex * this._rowHeight;

                    return true;
                }

                return false;
            },

            getSelectedRowIntoView: function (force) {
                return this._getRowIntoView(this._selectedIndex, force);
            },

            cacheRows: function (aboveRange, visibleRange, belowRange) {
            },

            _updateViewport: function (includeNonDirtyRows) {
                var visibleIndices, firstIndex, lastIndex, cachingStart, cachingEnd,
                resultCount = this._count, i, dataIndex, nodeState,
                lastVisible, above = [], below = [], visible = [],
                states = this._expandStates || [],
                maxIndex = this._expandedCount - 1;

                visibleIndices = this._getVisibleRowIndices();
                firstIndex = visibleIndices.first;
                lastIndex = visibleIndices.last;

                //expand viewport by 3 rows for smooth scrolling
                firstIndex = Math.max(0, firstIndex - this._options.extendViewportBy);
                lastIndex = Math.min(maxIndex, lastIndex + this._options.extendViewportBy);

                //make sure we are using all of our payload size
                cachingStart = Math.max(0, firstIndex - this._options.payloadSize);
                cachingEnd = Math.min(maxIndex, lastIndex + this._options.payloadSize);

                dataIndex = this._getDataIndex(cachingStart);
                lastVisible = firstIndex;

                for (i = cachingStart; i <= cachingEnd && dataIndex < resultCount; i++) {
                    if (i < firstIndex) {
                        above[above.length] = [i, dataIndex];
                    }
                    else if (i > lastIndex) {
                        below[below.length] = [i, dataIndex];
                    }
                    else {
                        visible[visible.length] = [i, dataIndex];
                        lastVisible = i;
                    }

                    nodeState = states[dataIndex]; //nodeState might be undefined
                    if (nodeState < 0) {
                        dataIndex += (1 - nodeState);
                    }
                    else {
                        dataIndex++;
                    }
                }

                this.cacheRows(above, visible, below);
                this._drawRows(visible, includeNonDirtyRows);
            },

            _cleanUpRows: function () {
                var rows = this._rows, row, dataIndex, hasGutter = this._gutter;

                for (dataIndex in rows) {
                    row = rows[dataIndex];
                    row.row.remove();
                    if (hasGutter) {
                        row.gutterRow.remove();
                    }
                }

                this._rows = {};
            },

            _drawRows: function (visibleRange, includeNonDirtyRows) {
                var i, l, array, rows = this._rows, row, rowIndex, dataIndex, newRows = {},
                states = this._expandStates, expandedState = 0, levels = this._indentLevels, level = 0,
                hasGutter = this._gutter, canvasDom = this._canvas[0], gutterCanvasDom,
                rowDom, gutterRowDom, updateRow, fragment, gutterFragment;

                fragment = document.createDocumentFragment();

                if (hasGutter) {
                    gutterCanvasDom = this._gutter[0];
                    gutterFragment = document.createDocumentFragment();
                }

                this._rows = newRows;

                for (i = 0, l = visibleRange.length; i < l; i++) {
                    array = visibleRange[i];
                    rowIndex = array[0];
                    dataIndex = array[1];

                    row = rows[dataIndex];
                    if (row) {
                        updateRow = (row.rowIndex !== rowIndex);
                        if (updateRow) {
                            row.rowIndex = rowIndex;
                        }
                        else {
                            updateRow = row.dirty;
                            delete row.dirty;
                        }

                        if (includeNonDirtyRows) {
                            updateRow = true;
                        }

                        delete rows[dataIndex];
                    }
                    else {
                        updateRow = true;
                        row = { rowIndex: rowIndex, dataIndex: dataIndex };
                        rowDom = domElem("div", "grid-row grid-row-normal");
                        rowDom.id = "row_" + this._getId() + "_" + rowIndex;
                        fragment.appendChild(rowDom);
                        row.row = $(rowDom).data("grid-row-info", row);

                        if (hasGutter) {
                            gutterRowDom = domElem("div", "grid-gutter-row grid-gutter-row-normal");
                            gutterFragment.appendChild(gutterRowDom);

                            row.gutterRow = $(gutterRowDom).data("grid-row-info", row);
                        }
                    }

                    newRows[dataIndex] = row;

                    if (updateRow) {
                        if (states) {
                            expandedState = states[dataIndex];
                            level = levels[dataIndex];
                        }

                        this._updateRow(row, rowIndex, dataIndex, expandedState, level);
                    }
                }

                for (dataIndex in rows) {
                    row = rows[dataIndex];

                    row.row.remove();
                    if (hasGutter) {
                        row.gutterRow.remove();
                    }
                }

                canvasDom.appendChild(fragment);
                if (hasGutter) {
                    gutterCanvasDom.appendChild(gutterFragment);
                }
            },

            updateRow: function (rowIndex, dataIndex) {
                var rowInfo, expandedState = 0, level = 0;

                if (typeof dataIndex === "undefined" || dataIndex < 0) {
                    dataIndex = this._getDataIndex(rowIndex);
                }
                else if (typeof rowIndex === "undefined" || rowIndex < 0) {
                    rowIndex = this._getRowIndex(dataIndex);
                }

                rowInfo = this._rows[dataIndex];

                if (rowInfo) {
                    if (this._expandStates) {
                        expandedState = this._expandStates[dataIndex];
                        level = this._indentLevels[dataIndex];
                    }

                    this._updateRow(rowInfo, rowIndex, dataIndex, expandedState, level);
                }
            },

            _updateRow: function (rowInfo, rowIndex, dataIndex, expandedState, level) {
                var row, rowElem, gutterOptions, gutterRow, gutterRowElem, gutterDropElem,
					gutterIconElem, gutterIconCss, gutterCheckboxCellElem, gutterCheckbox, indentIndex, i, l, columns, column, cellValue$;

                indentIndex = this._indentIndex;

                if (this._gutter) {
                    gutterOptions = this._options.gutter;

                    gutterRow = rowInfo.gutterRow;
                    gutterRowElem = gutterRow[0];
                    gutterRowElem.style.top = (rowIndex * this._rowHeight) + "px";
                    gutterRowElem.style.left = "0px";
                    gutterRowElem.style.width = (this._gutterWidth) + "px";
                    gutterRowElem.style.height = (this._rowHeight) + "px";

                    if (gutterOptions.contextMenu) {
                        gutterDropElem = domElem("div", "grid-gutter-cell grid-gutter-drop grid-gutter-menu");

                        // See if this row has a context menu.  If not remove the drop icon
                        if (this._options.source && this._options.source[dataIndex] && this._options.source[dataIndex].noContextMenu) {
                            $(gutterDropElem).removeClass('grid-gutter-drop');
                        }
                        gutterRow.append(gutterDropElem);
                    }

                    if (gutterOptions.checkbox) {
                        gutterCheckbox = $(domElem("input", "checkbox " + (gutterOptions.checkbox.cssClass || ""))).attr("type", "checkbox");
                        gutterCheckboxCellElem = domElem("div", "grid-gutter-cell grid-gutter-checkbox");
                        gutterCheckboxCellElem.appendChild(gutterCheckbox[0]);
                        gutterRowElem.appendChild(gutterCheckboxCellElem);
                    }

                    if (gutterOptions.icon) {
                        gutterIconCss = "grid-gutter-cell grid-gutter-icon ";

                        if (typeof gutterOptions.icon.cssClass !== "undefined") {
                            gutterIconCss += gutterOptions.icon.cssClass + " ";
                        }

                        if (typeof gutterOptions.icon.index !== "undefined") {
                            gutterIconCss += (this.getColumnValue(dataIndex, gutterOptions.icon.index, -1) || "") + " ";
                        }

                        if (gutterOptions.icon.ownerDraw !== false) {
                            gutterIconCss += (this._getGutterIconClass(rowIndex, dataIndex, expandedState, level) || "");
                        }

                        gutterIconElem = domElem("div", gutterIconCss);
                        gutterRowElem.appendChild(gutterIconElem);
                    }

                    this._drawGutterCell(rowInfo, rowIndex, dataIndex, expandedState, level);
                }

                row = rowInfo.row;
                row.empty();
                rowElem = row[0];
                rowElem.style.top = (rowIndex * this._rowHeight) + "px";
                rowElem.style.left = this._gutterWidth + "px";
                rowElem.style.height = (this._rowHeight) + "px";
                rowElem.style.width = isNaN(this._contentSize.width) ? "" : (this._contentSize.width + 2) + "px";


                columns = this._columns;

                for (i = 0, l = columns.length; i < l; i++) {
                    column = columns[i];
                    if (column.hidden) {
                        continue;
                    }
                    cellValue$ = column.getCellContents.apply(this, [rowInfo, dataIndex, expandedState, level, column, indentIndex, i]);
                    if (cellValue$) {
                        rowElem.appendChild(cellValue$[0]);
                    }
                }

                TFS.UI.makeElementUnselectable(rowElem);
                if (this._gutter) {
                    TFS.UI.makeElementUnselectable(gutterRowElem);
                }

                this._updateRowSelectionStyle(rowInfo, this._selectedRows, this._selectedIndex);

                // Let any listeners know the row was updated
                if (this._options.enabledEvents && Grid.EVENT_ROW_UPDATED in this._options.enabledEvents) {
                    // don't use this._fire because it bubbles events which shouldn't happen in this case.
                    this._element.triggerHandler(Grid.EVENT_ROW_UPDATED, [rowInfo]);
                }
            },

            _getGutterIconClass: function (rowIndex, dataIndex, expandedState, level) {
                return "";
            },

            _drawGutterCell: function (rowInfo, rowIndex, dataIndex, expandedState, level) {
            },

            _drawCell: function (rowInfo, dataIndex, expandedState, level, column, indentIndex, columnOrder) {
                /// <summary>Default implementation for creating the contents of a given cell.
                ///
                /// Custom Drawn Columns:
                /// If you want a custom drawn column, then the preferred method is to set a "getCellContents" property
                /// on the column to a function that takes the same parameters as this function and returns a jQuery
                /// object that represents the contents.
                ///
                /// </summary>
                /// <param name="rowInfo" type="Object">The information about grid row that is being rendered.</param>
                /// <param name="dataIndex" type="Number">The index of the row.</param>
                /// <param name="expandedState" type="Number">Number of children in the tree under this row recursively.</param>
                /// <param name="level" type="Number">The hierarchy level of the row.</param>
                /// <param name="column" type="Object">Information about the column that is being rendered.</param>
                /// <param name="indentIndex" type="Number">Index of the column that is used for the indentation.</param>
                /// <param name="columnOrder" type="Number">The display order of the column.</param>
                /// <returns>Returns jQuery element representing the requested grid cell. The first returned element will be appended
                /// to the row (unless the function returns <c>null</c> or <c>undefined</c>).</returns>
                var value,
                    treeSign,
                    indent,
                    cell,
                    cellDom,
                    width = column.width || 20,
                    href;

                cellDom = domElem("div", "grid-cell");
                cellDom.style.width = isNaN(width) ? width : width + "px";
                cell = $(cellDom);

                if (typeof column.hrefIndex !== "undefined") {
                    href = this.getColumnValue(dataIndex, column.hrefIndex, -1);
                }

                value = this.getColumnText(dataIndex, column, columnOrder);

                if (href) {
                    cell.append($("<a/>").attr("href", href).text(value));
                }
                else {
                    if (value) {
                        cell.text(value);
                    }
                    else {
                        // add non-breaking whitespace to ensure the cell has the same height as non-empty cells
                        cell.html("&nbsp;");
                    }
                }

                // Set the tooltip
                var tooltipText = this.getColumnText(dataIndex, column, columnOrder, true);
                if (tooltipText) {
                    var tooltip = { content: tooltipText.htmlEncode() };
                    cell.attr("data-plugin-vs-tooltip", JSON.stringify(tooltip));
                }

                if (columnOrder === indentIndex && level > 0) {
                    indent = ((level * 16) - 13); //TODO: magic numbers?
                    column.indentOffset = indent;
                    if (expandedState !== 0) {
                        treeSign = $(domElem("div", "icon grid-tree-icon")).appendTo(cell).css("left", indent);

                        if (expandedState > 0) {
                            treeSign.addClass("icon-tree-expanded");
                        }
                        else {
                            treeSign.addClass("icon-tree-collapsed");
                        }
                    }

                    cell.css("textIndent", (level * 16) + "px");
                }

                if (column.getCellCSSClass) {
                    var cellStyle = column.getCellCSSClass.call(this, dataIndex, column, columnOrder, this._dataSource);
                    if (cellStyle) {
                        cell.addClass(cellStyle);
                    }
                }

                if (column.rowCss) {
                    cell.addClass(column.rowCss);
                }

                return cell;
            },

            _drawHeader: function () {
                var i, l, column, columns = this._columns,
                    cell, cellDom, sepDom, sortDom, cellValue, separator, sort, sortOrder = this._sortOrder, fragment;

                if (this._header) {
                    fragment = document.createDocumentFragment();

                    for (i = 0, l = columns.length; i < l; i++) {
                        column = columns[i];
                        if (column.hidden) {
                            continue;
                        }

                        cellDom = domElem("div", "grid-header-column");
                        cell = $(cellDom);

                        // Set the tooltip
                        if (column.tooltip) {
                            var tooltip = { content: column.tooltip };
                            cell.attr("data-plugin-vs-tooltip", JSON.stringify(tooltip));
                        }

                        // Adjusting the width of the column
                        cellDom.style.width = (column.width || 20) + "px";
                        //cell.data("column-index", i);
                        cellDom._data = { columnIndex: i, header: true }; //jQuery.data accesses  which is slow on IE

                        // Creating the separator element for column resize
                        sepDom = domElem("div", "separator");
                        if (column.fixed) {
                            // Don't show resize cursor for fixed size columns
                            sepDom.style.cursor = "auto";
                        }
                        separator = $(sepDom);
                        sepDom._data = { columnIndex: i, separator: true }; //jQuery.data accesses DOM which is slow on IE
                        cellDom.appendChild(sepDom);
                        // Add an element for cell's value
                        cellValue = column.getHeaderCellContents.apply(this, [column]);

                        if (column.headerCss) {
                            cellValue.addClass(column.headerCss);
                        }

                        // TODO: Remove this when daytona fixes parent tooltip bug
                        if (column.tooltip) {
                            var tooltip = { content: column.tooltip };
                            cellValue.attr("data-plugin-vs-tooltip", JSON.stringify(tooltip));
                        }

                        cellDom.appendChild(cellValue && cellValue[0]);

                        // Creating the sort element for enabling the sort operation when it's clicked
                        sortDom = domElem("div", "sort-handle");
                        sort = $(sortDom);

                        $.each(sortOrder, function () {
                            if (this.index === column.index) {
                                if (this.order === "asc") {
                                    // Sorted asc
                                    cell.addClass("ascending");
                                }
                                else if (this.order === "desc") {
                                    // Sorted desc
                                    cell.addClass("descending");
                                }

                                return false;
                            }
                        });

                        cellDom.appendChild(sortDom);

                        fragment.appendChild(cellDom);
                    }

                    this._headerCanvas.empty();
                    this._columnMovingElement = null;
                    this._headerCanvas[0].appendChild(fragment);

                    TFS.UI.makeElementUnselectable(this._header[0]);
                }
            },

            _drawHeaderCellValue: function (column) {
                /// <summary>Default implementation for creating the element that represents content of a header cell.
                ///
                /// Custom Drawn Column Header:
                /// If you want a custom drawn column header, then the preferred method is to set a "getHeaderCellContents" property
                /// on the column to a function that takes the same parameters as this function and returns a jQuery
                /// object that represents the contents.
                ///
                /// </summary>
                /// <param name="column" type="Object">Information about the header column that is being rendered.</param>
                /// <returns>Returns jQuery element representing the requested header cell contents.</returns>
                var cell = $("<div/>").text(column.text || "").addClass("title");
                if (column.index === this._indentIndex && !(this._indentLevels === "undefined" || this._indentLevels === null)) {
                    cell.addClass("indented-title");
                }

                return cell;
            },

            _layoutContentSpacer: function () {
                var width = 0, height, i, l, columns = this._columns, scrollTop, scrollLeft;

                for (i = 0, l = columns.length; i < l; i++) {
                    if (columns[i].hidden) {
                        continue;
                    }
                    width += (columns[i].width || 20) + this._cellOffset;
                }

                // TODO: Magic number 2 here means 1px left border + 1px right border. Come up with a
                // better solution for this. We might set the box model to content-box but borders don't
                // fit very well in this case. If we don't apply this hack, cells don't fit in the row
                // and last cell breaks into next line.
                width = width + 2;
                height = Math.max(1, this._expandedCount * this._rowHeight); // we need horizontal scroll bar even if there is no result

                this._contentSpacer.width(width);
                this._contentSpacer.height(height);

                if (this._gutter) {
                    this._gutter.height(height);
                }

                this._ignoreScroll = true;
                try {
                    scrollTop = Math.max(0, Math.min(this._scrollTop, height - this._canvasHeight));

                    if (scrollTop !== this._scrollTop) {
                        this._scrollTop = scrollTop;
                        this._canvas[0].scrollTop = scrollTop;
                    }

                    scrollLeft = Math.max(0, Math.min(this._scrollLeft, width - this._canvasWidth));

                    if (scrollLeft !== this._scrollLeft) {
                        this._scrollLeft = scrollLeft;
                        this._canvas[0].scrollLeft = scrollLeft;
                    }
                } finally {
                    this._ignoreScroll = false;
                }

                this._contentSize.width = width;
                this._contentSize.height = height;
            },

            _layoutHeader: function () {
                if (this._header) {
                    this._headerCanvas.css("left", this._gutterWidth - this._scrollLeft);
                }

                if (this._gutter) {
                    this._gutter.css("left", this._scrollLeft);
                }
            },

            layout: function () {
                this._measureCanvasSize();
                this._cleanUpRows();
                this._fixScrollPos();
                this._layoutContentSpacer();
                this._updateViewport();

                this._layoutHeader();
                this._drawHeader();

                Diag.logTracePoint("Grid.layout.complete");
            },

            _fixScrollPos: function () {
                var oldIgnoreScroll = this._ignoreScroll;
                this._ignoreScroll = true;
                try {
                    this._canvas[0].scrollLeft = this._scrollLeft;
                    this._canvas[0].scrollTop = this._scrollTop;
                }
                finally {
                    this._ignoreScroll = oldIgnoreScroll;
                }
            },

            redraw: function () {
                this._fixScrollPos();
                this._redraw(true);
            },

            _redraw: function (includeNonDirtyRows) {
                this._layoutHeader();
                this._updateViewport(includeNonDirtyRows);
            },

            getColumnValue: function (dataIndex, columnIndex, columnOrder) {
                /// <summary>Gets the value for a column. The default use of the return value is to
                /// convert it to a string and set it as the cell's text value.</summary>
                /// <param name="dataIndex" type="int">The index for the row data in the data source</param>
                /// <param name="columnIndex" type="int">The index of the column's data in the row's data array</param>
                /// <param name="columnOrder" type="int">The index of the column in the grid's column array. This is the current visible order of the column</param>
                return this._dataSource[dataIndex][columnIndex];
            },

            getColumnText: function (dataIndex, column, columnOrder, isForTooltip) {
                var value, text;
                
                value = column.getColumnValue.call(this, dataIndex, column.index, columnOrder, this._dataSource, isForTooltip);

                if (typeof value !== "string") {
                    text = CoreUtils.convertValueToDisplayString(value, column.format);
                }
                else {
                    text = value;
                }

                column.maxLength = Math.max(column.maxLength || 0, text.length);

                return text;
            },

            _getExpandState: function (dataIndex) {
                var result = 0;
                if (this._expandStates) {
                    if (typeof (this._expandStates[dataIndex] === "number")) {
                        result = this._expandStates[dataIndex];
                    }
                }
                return result;
            },

            /* selection */
            _selectRow: function (rowIndex, dataIndex, options) {
                var ctrl = options && options.ctrl,
                    shift = options && options.shift,
                    rightClick = options && options.rightClick;

                if (ctrl) {
                    // If ctrl key is pressed, selecting or deselecting only the row at rowIndex
                    this._addSelection(rowIndex, dataIndex, { toggle: true });
                }
                else if (shift) {
                    // If shift key is pressed, selecting the rows starting from selection start until the row at rowIndex
                    this._clearSelection();
                    this._addSelectionRange(rowIndex, dataIndex);
                }
                else if (rightClick) {
                    if (!this._selectedRows || !(rowIndex in this._selectedRows)) {
                        // Right-clicked a previously unselected row, select that row
                        this._clearSelection();
                        this._addSelection(rowIndex, dataIndex);
                    }
                    else {
                        // Right-clicked a previously selected row. Just update the selection index.
                        this._selectedIndex = rowIndex;
                        this._updateAriaAttribute();
                    }
                }
                else {
                    // Just selecting the single row at rowIndex
                    this._clearSelection();
                    this._addSelection(rowIndex, dataIndex);
                }
            },

            _selectAll: function () {
                if (this._count > 0 && this._options.allowMultiSelect !== false) {

                    // Clearing the selection first
                    this._clearSelection();
                    this._selectionStart = 0;

                    // Saving the selected rowIndex
                    var prevIndex = Math.max(0, this._selectedIndex);
                    this._addSelectionRange(this._count - 1, undefined, { doNotFireEvent: true });

                    // Restoring the selected rowIndex
                    this._selectedIndex = prevIndex;

                    this._updateSelectionStyles();
                    this._selectionChanged();
                }
            },

            getSelectedRowIndex: function () {
                return this._selectedIndex;
            },

            setSelectedRowIndex: function (selectedRowIndex) {
                this._clearSelection();
                this._addSelection(selectedRowIndex);
            },

            getSelectedDataIndex: function () {
                return this._getDataIndex(this._selectedIndex);
            },

            getSelectedDataIndices: function () {
                var index,
                    rows = this._selectedRows,
                    indices = [];

                if (rows) {
                    for (index in rows) {
                        indices[indices.length] = rows[index];
                    }
                }
                return indices;
            },

            ensureDataIndexExpanded: function (dataIndex) {
                /// <summary>Ensures that an item (identified by a data index) has an associated row by
                /// expanding any enclosing collapsed rows. Returns the rowIndex of the associated row.</summary>
                /// <param name="dataIndex" type="Number">The data index of the item to ensure is expanded</param>
                /// <returns type="Number" />
                Diag.assertParamIsNumber(dataIndex, "dataIndex");

                var rowIndex = this._getRowIndex(dataIndex);
                while (rowIndex < 0 || (dataIndex > 0 && rowIndex === 0)) {
                    this.expandNode(this._getDataIndex(-rowIndex));
                    rowIndex = this._getRowIndex(dataIndex);
                }
                return rowIndex;
            },

            setSelectedDataIndex: function (dataIndex, expandNodes) {
                /// <summary>Sets the selected item in the grid by the data index.
                /// Optionally ensure that the item is not hidden by collapsed rows.</summary>
                /// <param name="dataIndex" type="Number">The data index of item to show</param>
                /// <param name="expandNodes" type="Boolean">If <true>, all containing collapsed nodes will be expanded</param>
                Diag.assertParamIsNumber(dataIndex, "dataIndex");

                var rowIndex = expandNodes ? this.ensureDataIndexExpanded(dataIndex) : this._getRowIndex(dataIndex);
                this.setSelectedRowIndex(rowIndex);
            },

            _clearSelection: function () {
                this._selectionCount = 0;
                this._selectedRows = null;
            },

            _addSelection: function (rowIndex, dataIndex, options) {
                /// <summary>Highlights the row at the specified rowIndex</summary>
                /// <param name="rowIndex" type="Integer">Index of the row in the visible source (taking the expand/collapse states into account)</param>
                /// <param name="dataIndex" type="Integer">Index of the row in the overall source</param>
                /// <param name="options" type="Boolean">Specifies options such as:
                ///     - keepSelectionStart: Keepd the rowIndex as the basis for range selection
                ///     - doNotFireEvent: Prevents firing events
                ///     - toggle: Toggles the row in the selection

                var add,
                    keepSelectionStart = options && options.keepSelectionStart,
                    doNotFireEvent = options && options.doNotFireEvent,
                    toggle = options && options.toggle;

                if (this._options.allowMultiSelect === false) {
                    keepSelectionStart = false;
                    this._clearSelection();
                }

                if (!this._selectedRows) {
                    this._selectedRows = {};
                }

                if (rowIndex >= 0) {
                    add = true;

                    if (!(rowIndex in this._selectedRows)) {
                        // If not selected before increasing selection count
                        this._selectionCount++;
                    }
                    else if (toggle) {
                        // If the row already exists in the selection and toggle is enabled
                        // removing it from the selection
                        add = false;
                        this._selectionCount = Math.max(0, this._selectionCount - 1);
                        delete this._selectedRows[rowIndex];
                    }

                    if (typeof (dataIndex) !== "number") {
                        // If the dataIndex is not specified, finding it by using visible rowIndex
                        dataIndex = this._getDataIndex(rowIndex);
                    }

                    if (add) {
                        this._selectedRows[rowIndex] = dataIndex;
                    }

                    this._selectedIndex = rowIndex;
                    this._updateAriaAttribute();


                    if (this._selectionStart < 0 || !keepSelectionStart) {
                        this._selectionStart = rowIndex;
                    }
                }
                else {
                    dataIndex = -1;
                    this._selectedIndex = -1;
                }

                if (!doNotFireEvent) {
                    this._updateSelectionStyles();
                    this._selectionChanged();

                    this._selectedIndexChanged(this._selectedIndex, dataIndex);
                }
            },

            _addSelectionRange: function (rowIndex, dataIndex, options) {
                /// <summary>Highlights the rows beginning from the selection start until the row at the specified rowIndex</summary>
                /// <param name="rowIndex" type="Integer">Index of the row in the visible source (taking the expand/collapse states into account)</param>
                /// <param name="dataIndex" type="Integer">Index of the row in the overall source</param>

                var start,
                    end,
                    i,
                    nodeState,
                    doNotFireEvent = options && options.doNotFireEvent,
                    prevSelectedDataIndex = -1,
                    selectedDataIndex;

                if (this._options.allowMultiSelect === false) {
                    this._addSelection(rowIndex, dataIndex);
                }
                else {
                    if (this._selectedRows) {
                        prevSelectedDataIndex = this._selectedRows[this._selectedIndex];
                    }

                    if (this._selectionStart < 0) {
                        this._selectionStart = rowIndex;
                    }

                    start = Math.min(this._selectionStart, rowIndex);
                    end = Math.max(this._selectionStart, rowIndex);

                    if (typeof (dataIndex) !== "number" || start !== rowIndex) {
                        // If the dataIndex is not specified or rowIndex is different than start,
                        // finding it by using visible rowIndex
                        dataIndex = this._getDataIndex(start);
                    }

                    for (i = start; i <= end; i++) {
                        this._addSelection(i, dataIndex, { keepSelectionStart: true, doNotFireEvent: true });
                        if (i === rowIndex) {
                            selectedDataIndex = dataIndex;
                        }
                        nodeState = this._getExpandState(dataIndex);
                        if (nodeState < 0) {
                            dataIndex += (1 - nodeState);
                        }
                        else {
                            dataIndex++;
                        }
                    }

                    // Setting selected index to index of last selected row
                    this._selectedIndex = rowIndex;
                    this._updateAriaAttribute();

                    if (!doNotFireEvent) {
                        this._updateSelectionStyles();
                        this._selectionChanged();

                        if (prevSelectedDataIndex !== selectedDataIndex) {
                            this._selectedIndexChanged(this._selectedIndex, selectedDataIndex);
                        }
                    }
                }
            },

            _updateAriaAttribute: function () {
                /// <summary>This is especially necessary for screen readers to read each
                /// row when the selection changes. </summary>
                var rowInfo, dataIndex, id;

                // Updating only if the grid is active
                if (this._active) {
                    dataIndex = this._getDataIndex(this._selectedIndex);
                    if (dataIndex >= 0) {
                        // Getting row info using data index
                        rowInfo = this.getRowInfo(dataIndex);
                        if (rowInfo && rowInfo.row) {
                            id = rowInfo.row.attr("id");
                            if (id !== this._activeAriaId) {
                                // Setting active element attribute
                                this._focus.attr("aria-activedescendant", id);
                                this._activeAriaId = id;
                            }
                        }
                    }
                }
            },

            _updateSelectionStyles: function () {

                this.delayExecute("updateSelectionStyles", 10, true, function () {
                    var dataIndex, selectedRows = this._selectedRows,
                    focusIndex = this._selectedIndex,
                    rows = this._rows, rowInfo;

                    for (dataIndex in rows) {
                        rowInfo = rows[dataIndex];
                        this._updateRowSelectionStyle(rowInfo, selectedRows, focusIndex);
                    }
                });
            },

            _selectionChanged: function () {
                this.selectionChanged(this._selectedIndex, this._selectionCount, this._selectedRows);

                this._fire("selectionchanged", { selectedIndex: this._selectedIndex, selectedCount: this._selectionCount, selectedRows: this._selectedRows });
            },

            selectionChanged: function (selectedIndex, selectedCount, selectedRows) {
            },

            _selectedIndexChanged: function (selectedRowIndex, selectedDataIndex) {
                this.selectedIndexChanged(selectedRowIndex, selectedDataIndex);

                this._fire(Grid.EVENT_SELECTED_INDEX_CHANGED, [selectedRowIndex, selectedDataIndex]);
            },

            selectedIndexChanged: function (selectedRowIndex, selectedDataIndex) {

            },

            _updateRowSelectionStyle: function (rowInfo, selectedRows, focusIndex) {

                var rowIndex, row, gutterRow, active;

                rowIndex = rowInfo.rowIndex;
                row = rowInfo.row;
                gutterRow = rowInfo.gutterRow;
                active = this._active;

                row.removeClass("grid-row-selected grid-row-selected-blur grid-row-current");
                if (gutterRow) {
                    gutterRow.removeClass("grid-gutter-row-selected grid-gutter-row-selected-blur grid-gutter-row-current");
                    gutterRow.find("input.checkbox").attr("checked", false);
                }

                if (selectedRows && rowIndex in selectedRows) {
                    if (gutterRow) {
                        gutterRow.find("input.checkbox").attr("checked", true);
                    }

                    if (active) {
                        row.addClass("grid-row-selected");

                        if (gutterRow) {
                            gutterRow.addClass("grid-gutter-row-selected");
                        }
                    }
                    else {
                        row.addClass("grid-row-selected-blur");

                        if (gutterRow) {
                            gutterRow.addClass("grid-gutter-row-selected-blur");
                        }
                    }
                }

                if (rowIndex === focusIndex) {
                    row.addClass("grid-row-current");

                    if (gutterRow) {
                        gutterRow.addClass("grid-gutter-row-current");
                    }
                }
            },

            focus: function (timeout) {
                TFS.UI.tryFocus(this._focus, timeout);
            },

            /* Context menu support */
            _showContextMenu: function (rowInfo) {
                var gutter,
                    item,
                    menuOptions;

                menuOptions = this._options.contextMenu;

                if (!menuOptions) {
                    return;
                }

                if (rowInfo) {
                    gutter = rowInfo.gutterRow; // Getting the reference to the clicked gutter
                    item = this._dataSource[rowInfo.dataIndex];

                    menuOptions = $.extend({}, menuOptions, {
                        contextInfo: {
                            rowInfo: rowInfo,
                            item: item
                        }
                    });

                    if (!item.root) {
                        if (menuOptions.items) {
                            if ($.isFunction(menuOptions.items)) {
                                menuOptions.items = menuOptions.items.call(this, menuOptions.contextInfo);
                            }
                            else {
                                menuOptions.items = menuOptions.items.slice(0); //create a clone
                            }
                        }
                        else {
                            menuOptions.items = [];
                        }

                        if (this._popupMenu) {
                            this._popupMenu.dispose();
                            this._popupMenu = null;
                        }

                        this._popupMenu = this._createContextMenu(rowInfo, menuOptions);

                        // Displaying the popup
                        // Grid set tries to set focus on container mouse down event with a timeout
                        // This behavior causes our popup menu item to close immediately since it loses focus.
                        // Lets popup our menu in another epoch
                        Core.delay(this, 10, function () {
                            this._popupMenu.popup(this._focus[0], gutter);
                        });
                    }
                }
            },

            _createContextMenu: function (rowInfo, menuOptions) {
                /// <summary>Creates the context menu options. This function is intended to be overriden by derived objects.</summary>
                /// <param name="rowInfo" type="Object">The information about the row with context</param>
                /// <param name="menuOptions" type="Object">The menu information. See _createContextPopupMenuControl</param>
                /// <returns type="PopupMenu" />
                Diag.assertParamIsObject(rowInfo, "rowInfo");
                Diag.assertParamIsObject(menuOptions, "menuOptions");

                return this._createContextPopupMenuControl(menuOptions);
            },

            _createContextPopupMenuControl: function (menuOptions) {
                /// <summary>
                ///     Creates the PopupMenu control that houses the context menu items for the Grid. Note: this is intentionally
                ///     abstracted from _createContextMenu to allow directly calling it from deep derivations and avoiding inheritance
                ///     base propagation.
                /// </summary>
                /// <param name="menuOptions" type="Object">
                ///     The menu information:
                ///     {
                ///         contextInfo: { item, rowInfo}
                ///         items: the list of menu items
                ///     }
                /// </param>
                /// <returns type="PopupMenu" />
                Diag.assertParamIsObject(menuOptions, "menuOptions");

                var menuItems = $.map(menuOptions.items || [], function (menuItem) {
                    return menuItem;
                });

                return MenuControls.PopupMenu.createIn(this._element, $.extend(
                    {
                        align: "left-bottom"
                    },
                    menuOptions,
                    {
                        items: [{ childItems: MenuControls.sortMenuItems(menuItems) }]
                    }));
            },

            /* Events */

            _onContainerMouseDown: function (e) {
                this.focus(10);
            },

            _measureCanvasSize: function () {
                this._canvasHeight = this._canvas[0].clientHeight;
                this._canvasWidth = this._canvas[0].clientWidth;
            },

            _onContainerResize: function (e) {
                this.layout();
            },

            _setupDragEvents: function () {
                this._bind(window.document, "mousemove", delegate(this, this._onDocumentMouseMove), true);
                this._bind(window.document, "mouseup", delegate(this, this._onDocumentMouseUp), true);
            },

            _clearDragEvents: function () {
                this._unbind(window.document, "mousemove", null, true);
                this._unbind(window.document, "mouseup", null, true);
            },

            _onDocumentMouseMove: function (e) {
                var delta, column, newWidth,
                    columnSizing = this._columnSizing,
                    columnMoving = this._columnMoving;

                // Checking whether column sizing has started or not
                if (columnSizing && columnSizing.active === true) {
                    delta = e.pageX - columnSizing.origin;
                    newWidth = Math.max(15, columnSizing.originalWidth + delta);
                    column = this._columns[columnSizing.index];
                    column.width = newWidth;

                    this._applyColumnSizing(columnSizing.index);
                    this._moveSizingElement(columnSizing.index);
                }

                if (columnMoving && columnMoving.state > 0) {
                    delta = e.pageX - columnMoving.origin;
                    if (delta !== 0) {
                        columnMoving.state = 2;
                        this._moveColumnMovingElement(columnMoving.index, columnMoving.left + delta);
                    }
                }
            },

            _onDocumentMouseUp: function (e) {
                Core.delay(this, 0, function () {
                    this._tryFinishColumnSizing(false);
                    this._tryFinishColumnMoving(false);
                    this._clearDragEvents();
                });

                return false;
            },

            _onHeaderMouseDown: function (e) {
                var columnIndex, column, left, i, separator, headerColumn;

                // We should support header operations only for left mouse button
                if (e.which !== 1) {
                    return;
                }
                separator = $(e.target).closest(".separator");

                if (separator.length > 0) {
                    columnIndex = separator[0]._data.columnIndex;
                    column = this._columns[columnIndex];
                    if (!column.fixed) {
                        this._columnSizing = { active: true, index: columnIndex, originalWidth: column.width, origin: e.pageX };
                        this._moveSizingElement(columnIndex);

                        this._setupDragEvents();
                        return false;
                    }
                }
                else {
                    headerColumn = $(e.target).closest(".grid-header-column");

                    if (headerColumn.length > 0) {
                        columnIndex = headerColumn[0]._data.columnIndex;
                        column = this._columns[columnIndex];
                        if (this._options.allowMoveColumns && !column.fixed && column.canMove) {

                            this._columnMoving = { state: 1, index: columnIndex, width: column.width, origin: e.pageX };
                            left = 0;
                            i = 0;
                            while (i < columnIndex) {
                                column = this._columns[i++];
                                if (!column.hidden) {
                                    left += column.width;
                                }
                            }

                            this._columnMoving.left = left;

                            this._setupDragEvents();
                            return false;
                        }
                    }
                }
            },

            _onHeaderMouseUp: function (e) {
            },

            _onHeaderClick: function (e) {
                var columnIndex, column, headerColumn, separator;

                headerColumn = $(e.target).closest(".grid-header-column");

                if (headerColumn.length > 0) {
                    if (!this._columnSizing) {

                        separator = $(e.target).closest(".separator");

                        if (separator.length > 0) {
                            return false;
                        }
                        else {
                            columnIndex = headerColumn[0]._data.columnIndex;
                            column = this._columns[columnIndex];
                            if (column.canSortBy && !column.fixed) {
                                this._sortBy(column, e.shiftKey);
                            }
                        }
                    }
                }
            },

            _onHeaderDblClick: function (e) {
                // Determining whether the element is a separator
                var separator, columnIndex, column,
                    maxLength, ratio, originalWidth;

                separator = $(e.target).closest(".separator");

                if (separator.length > 0) {

                    // Cancel any pending sizing/moving events triggered by the double-clicks first mouse click
                    this._tryFinishColumnSizing(true);
                    this._tryFinishColumnMoving(true);

                    columnIndex = separator[0]._data.columnIndex;
                    column = this._columns[columnIndex];

                    maxLength = Math.max(column.maxLength || 0, 3);
                    ratio = 1.1 + 0.7 * Math.exp(-maxLength / 20);
                    originalWidth = column.width;
                    column.width = (column.indentOffset || 0) + Math.round(maxLength * ratio * this._unitEx);
                    this._applyColumnSizing(columnIndex, originalWidth, true);

                    return false;
                }
            },

            /* Column operations like resize, move, sort */
            _moveSizingElement: function (columnIndex) {
                var canvas = this._canvas,
                    canvasDom = canvas[0],
                    sizingElement = this._sizingElement,
                    column, i = 0, left = this._gutterWidth;

                if (!sizingElement) {
                    // If there is no sizing element around, creating one
                    if (columnIndex < 0) {
                        return;
                    }

                    sizingElement = $(domElem("div", "grid-column-sizing")).appendTo(canvas);
                    this._sizingElement = sizingElement;
                }

                sizingElement.height(canvasDom.clientHeight - 1);
                sizingElement.css("top", this._scrollTop);

                if (columnIndex < 0) {
                    sizingElement.css("left", -5000).css("top", -5000).height(0);
                }
                else {
                    while (i <= columnIndex) {
                        column = this._columns[i++];
                        if (!column.hidden) {
                            left += column.width;
                        }
                    }

                    sizingElement.css("left", left - 1);
                }
            },

            _getVisibleColumnIndex: function (columnIndex) {
                /// <summary>
                ///     Given a column index will provide the visible index of this column. That is, it will take in to consideration any
                ///     hidden columns and omit them from the index count.
                /// </summary>
                /// <param name="columnIndex" type="Number">The 0-based global column index</param>
                /// <returns type="Number">The 0-based visible column index</returns>
                Diag.assertParamIsNumber(columnIndex, "columnIndex");

                var columnCounter = 0,
                    visibleColumnIndex = 0,
                    length = this._columns.length;

                Diag.assert(columnIndex < length, "Index out of array bounds");

                if (this._columns[columnIndex].hidden) {
                    return -1; // If the column is hidden then it has no visible index
                }

                while (columnCounter < columnIndex) {
                    if (!this._columns[columnCounter].hidden) {
                        visibleColumnIndex++;
                    }
                    columnCounter++;
                }

                return visibleColumnIndex;
            },

            _applyColumnSizing: function (columnIndex, initialWidth, finish) {

                var domColumnIndex = this._getVisibleColumnIndex(columnIndex) + 1, // Add 1 because DOM nth-child selector use 1-based (not 0-based) indices
                    column = this._columns[columnIndex],
                    columnSizeChanged = false;

                initialWidth = initialWidth || -1;

                if (column) {
                    columnSizeChanged = column.width !== initialWidth;

                    $(".grid-header-column:nth-child(" + domColumnIndex + ")", this._headerCanvas).width(column.width);
                }

                if (finish === true) {
                    if (columnSizeChanged) {
                        this.layout();
                    }

                    this._onColumnResize(column);
                }
            },

            _onColumnResize: function (column) {
                this._fire("columnresize", [column]);
            },

            _tryFinishColumnSizing: function (cancel) {
                var columnSizing = this._columnSizing;
                if (columnSizing) {
                    if (columnSizing.active === true) {
                        if (!cancel) {
                            this._applyColumnSizing(columnSizing.index, columnSizing.originalWidth, true);
                        }
                        this._moveSizingElement(-1);
                    }

                    this._columnSizing = null;
                }
            },

            _moveColumnMovingElement: function (columnIndex, left) {
                var headerCanvas = this._headerCanvas,
                    canvas = this._canvas,
                    columnMovingElement = this._columnMovingElement,
                    columnMovingPinElement = this._columnMovingPinElement,
                    column, len = this._columns.length, lf = this._gutterWidth, i = 0, insertIndex = 0, pos;

                if (!columnMovingElement) {
                    if (columnIndex < 0) {
                        return;
                    }

                    columnMovingElement = $(domElem("div", "grid-column-moving")).appendTo(headerCanvas);
                    this._columnMovingElement = columnMovingElement;

                    columnMovingPinElement = $(domElem("div", "grid-column-moving-placer")).appendTo(canvas)
                        .height(4).css("left", -500).css("top", -500);
                    this._columnMovingPinElement = columnMovingPinElement;
                }

                if (columnIndex < 0) {
                    columnMovingElement.css("left", -500000);
                    columnMovingPinElement.css("left", -500).css("top", -500);
                }
                else {
                    column = this._columns[columnIndex];
                    columnMovingElement.width(column.width)
                        .css("left", left)
                        .css("top", 0)
                        .text(column.text);

                    pos = left + (column.width >> 1);
                    while (i < len) {
                        column = this._columns[i];
                        if (!column.hidden) {
                            if ((pos - lf) > (column.width / 2)) {
                                insertIndex = i + 1;
                            }
                            else {
                                break;
                            }

                            lf += column.width;
                        }
                        i++;
                    }

                    if (insertIndex < columnIndex || insertIndex > (columnIndex + 1)) {
                        this._columnInsert = insertIndex;
                        columnMovingPinElement.css("left", lf - 4).css("top", this._scrollTop);
                    }
                    else {
                        this._columnInsert = -1;
                        columnMovingPinElement.css("left", -500).css("top", -500);
                    }
                }
            },

            _applyColumnMoving: function (sourceIndex, targetIndex) {
                var sourceColumn, targetColumn,
                    columns = this._columns;

                if (targetIndex >= 0) {

                    if (targetIndex > sourceIndex) {
                        targetIndex--;
                    }

                    sourceColumn = columns[sourceIndex];
                    targetColumn = columns[targetIndex];

                    if (this._options.allowMoveColumns && !targetColumn.fixed && targetColumn.canMove) {
                        // Moving the column in the columns array
                        columns.splice(sourceIndex, 1);
                        columns.splice(targetIndex, 0, sourceColumn);

                        this._determineIndentIndex();
                        // Updating UI so that the header and rows are rendered according to the new column order
                        this.layout();
                        this._onColumnMove(sourceIndex, targetIndex);
                    }
                }
            },

            _onColumnMove: function (sourceIndex, targetIndex) {
                this._fire("columnmove", [sourceIndex, targetIndex]);
            },

            _tryFinishColumnMoving: function (cancel) {
                var columnMoving = this._columnMoving;
                if (columnMoving) {
                    if (columnMoving.state > 1) {
                        this._moveColumnMovingElement(-1);
                        if (!cancel) {
                            this._applyColumnMoving(columnMoving.index, this._columnInsert);
                        }
                    }

                    this._columnMoving = null;
                }
            },

            _getSortColumns: function (sortOrder) {
                var columns = this._columns, c, i, l, sc, sortColumns = [];

                for (i = 0, l = sortOrder.length; i < l; i++) {
                    sc = sortOrder[i];
                    $.each(columns, function () {
                        if (this.index === sc.index) {
                            c = this;
                            return false;
                        }
                    });

                    sortColumns.push(c);
                }

                return sortColumns;
            },

            _sortBy: function (column, add) {
                var sortOrder = this._sortOrder.slice(0), i, l, sc, found = false, sortColumns;

                if (column) {
                    for (i = 0, l = sortOrder.length; i < l; i++) {
                        sc = sortOrder[i];
                        if (sc.index === column.index) {
                            sortOrder.splice(i, 1);
                            found = true;
                            break;
                        }
                    }

                    if (found) {
                        sc = { index: sc.index, order: sc.order === "asc" ? "desc" : "asc" };
                    }
                    else {
                        sc = { index: column.index, order: "asc" };
                    }

                    if (add && this._options.allowSortOnMultiColumns) {
                        sortOrder.push(sc);
                    }
                    else {
                        sortOrder = [sc];
                    }
                }

                sortColumns = this._getSortColumns(sortOrder);

                this._onSort(sortOrder, sortColumns);
            },

            _onSort: function (sortOrder, sortColumns) {
                if (this.onSort(sortOrder, sortColumns) !== false) {
                    this._fire("sort", [{ sortOrder: sortOrder, sortColumns: sortColumns }]);
                }
            },

            onSort: function (sortOrder, sortColumns) {
                if (this._options.autoSort) {
                    this._trySorting(sortOrder, sortColumns);
                    this._sortOrder = sortOrder;
                    this.layout();
                }

                // Refresh UI after the sort
                TFS.Diag.logTracePoint('Grid.OnSort.Complete');
            },

            _trySorting: function (sortOrder, sortColumns) {
                var that = this;

                function defaultComparer(column, order, rowA, rowB) {
                    var v1 = rowA[column.index],
                        v2 = rowB[column.index];

                    if (typeof v1 === "undefined" || v1 === null) {
                        if (typeof v2 === "undefined" || v2 === null) {
                            return 0;
                        }
                        else {
                            return -1;
                        }
                    }

                    return String.localeIgnoreCaseComparer(v1, v2);
                }

                if (!sortColumns) {
                    sortColumns = this._getSortColumns(sortOrder);
                }

                function sortComparer(rowA, rowB) {
                    var i, c, result, column, comparer;
                    for (i = 0; i < sortOrder.length; i++) {
                        c = sortOrder[i];
                        column = sortColumns[i];
                        comparer = column.comparer || defaultComparer;
                        result = comparer.call(that, column, c.order, rowA, rowB);

                        if (result === 0) {
                            continue;
                        }
                        else if (c.order === "desc") {
                            return -result;
                        }
                        else {
                            return result;
                        }
                    }

                    return 0;
                }

                // Archive the selected datas and rows before sorting,
                // as we will try to recover them after sorting.
                var selectedDatas = [],
                    sumOfSelectedRowIndices = 0;

                for (var rowIndex in this._selectedRows) {
                    var dataIndex = this._selectedRows[rowIndex];
                    var data = this._dataSource[dataIndex];
                    if (data) {
                        selectedDatas.push(data);
                        sumOfSelectedRowIndices += parseInt(rowIndex);
                    }
                }

                // Convert the arrays to tree as we only sort the siblings
                var rootNode = new TreeNode(null, null);
                addItemsToTree(this._dataSource, this._expandStates, 0, this._dataSource.length, rootNode);

                // Sort the siblings
                if (this._sortOrder
                    && this._sortOrder.length === 1
                    && sortOrder.length === 1
                    && this._sortOrder[0] !== sortOrder[0] // we couldn't use this optimization if the input sortOrder is same object as this._sortOrder
                    && this._sortOrder[0].index === sortOrder[0].index) {
                    // If only the ordering is different, we just need to reverse the siblings, which is much faster than sort
                    if (this._sortOrder[0].order !== sortOrder[0].order) {
                        CoreUtils.walkTree.call(rootNode, function(node) {
                            Array.stableReverse(node.children, function(v1, v2) {
                                return sortComparer(v1.data, v2.data);
                            });
                        });
                    }
                } else {
                    CoreUtils.walkTree.call(rootNode, function(node) {
                        node.children.sort(function(v1, v2) {
                            return sortComparer(v1.data, v2.data);
                        });
                    });
                }

                // Convert the sorted tree back to arrays
                // Note the root node is excluded from the array as it's used only to build up the tree struct
                this._dataSource = [];
                this._expandStates = [];
                for (var i = 0; i < rootNode.children.length; i++) {
                    CoreUtils.walkTree.call(rootNode.children[i], function(node) {
                        that._dataSource.push(node.data);
                        that._expandStates.push(node.expandState);
                    });
                }

                this._indentLevels = expand(this._expandStates);
                this._updateRanges();

                // Recover the selected rows, which means the objects selected before sorting should be selected after sorting
                if (this._selectionCount > 0) {
                    this._clearSelection();

                    var sumOfNewSelectedRowIndices = 0;
                    for (var i = 0; i < selectedDatas.length; i++) {
                        var dataIndex = this._dataSource.indexOf(selectedDatas[i]);
                        if (dataIndex >= 0) {
                            var rowIndex = this._getRowIndex(dataIndex);
                            this._addSelection(rowIndex, dataIndex);
                            sumOfNewSelectedRowIndices += rowIndex;
                        }
                    }

                    // Update scroller's top to ensure the selected row is still visible at the same offset of the screen.
                    // In case there are multiple selections, use the average row index to calculate the offset.
                    // TFS Grid will validate the scroller's top before redraw the rows, so we don't need to worry
                    // about if this scroller's top exceeds the boundary.
                    this._scrollTop += (sumOfNewSelectedRowIndices - sumOfSelectedRowIndices) / this._selectionCount * this._rowHeight;
                }
            },
            /* End of Column operations like resize, move, sort */

            _onCanvasScroll: function (e) {
                var canvas = this._canvas[0];
                this._resetScroll = true;
                this._scrollLeft = canvas.scrollLeft;
                this._scrollTop = canvas.scrollTop;

                if (!this._ignoreScroll) {
                    this._redraw();
                }

                return false;
            },

            _handleEvent: function (e, handler, eventName, args) {
                var dataIndex, rowIndex = this._selectedIndex, eventArgs;
                dataIndex = this._getDataIndex(rowIndex);

                eventArgs = { rowIndex: rowIndex, dataIndex: dataIndex, rowInfo: this._rows[dataIndex], event: e };

                if (args) {
                    $.extend(eventArgs, args);
                }

                if (!handler || handler.call(this, eventArgs) !== false) {
                    return this._fire(eventName, eventArgs);
                }
                else {
                    return false;
                }
            },

            _getRowInfoFromEvent: function (e, selector) {
                return $(e.target).closest(selector).data("grid-row-info");
            },

            _onRowMouseDown: function (e) {
                var rowInfo = this._getRowInfoFromEvent(e, ".grid-row");

                if (rowInfo) {
                    if ($(e.target).hasClass("grid-tree-icon") && e.which === 1) { // Left mouse button
                        this._onToggle(rowInfo);
                    }
                    else {
                        this._selectRow(rowInfo.rowIndex, rowInfo.dataIndex, {
                            ctrl: e.ctrlKey,
                            shift: e.shiftKey,
                            rightClick: e.which === 3
                        });
                        return this._handleEvent(e, this.onRowMouseDown, "rowmousedown", {
                            row: rowInfo.row,
                            rowInfo: rowInfo,
                            rowIndex: rowInfo.rowIndex,
                            dataIndex: rowInfo.dataIndex
                        });
                    }
                }
            },

            onRowMouseDown: function (eventArgs) {
            },

            _onRowClick: function (e) {
                var rowInfo = this._getRowInfoFromEvent(e, ".grid-row");

                if (rowInfo) {
                    return this._handleEvent(e, this.onRowClick, "rowclicked", {
                        row: rowInfo.row,
                        rowInfo: rowInfo,
                        rowIndex: rowInfo.rowIndex,
                        dataIndex: rowInfo.dataIndex
                    });
                }
            },

            onRowClick: function (eventArgs) {
                if (this._rowClickCallback !== null) {
                    this._rowClickCallback(this._dataSource[eventArgs.dataIndex]);
                }
            },

            _onRowDoubleClick: function (e) {
                var rowInfo = this._getRowInfoFromEvent(e, ".grid-row");

                if (rowInfo) {
                    if ($(e.target).hasClass("grid-tree-icon")) {
                        // Do nothing.  Let _onRowMouseDown handle this.
                    }
                    else {
                        if (this._handleEvent(e, this.onRowDoubleClick, "rowdblclick", {
                            row: rowInfo.row,
                            rowInfo: rowInfo,
                            rowIndex: rowInfo.rowIndex,
                            dataIndex: rowInfo.dataIndex
                        }) !== false) {
                            return this._onOpenRowDetail(e, { rowIndex: rowInfo.rowIndex, dataIndex: rowInfo.dataIndex });
                        }
                        else {
                            return false;
                        }
                    }
                }
            },

            onRowDoubleClick: function (eventArgs) {

            },

            _onGutterClick: function (e) {
                var rowInfo = this._getRowInfoFromEvent(e, ".grid-gutter-row"), target;

                if (rowInfo) {
                    if (this._options.gutter.checkbox) {
                        target = $(e.target);
                        if (target.hasClass("checkbox")) {
                            this._addSelection(rowInfo.rowIndex, rowInfo.dataIndex, { toggle: true });
                        }
                    }
                    else {
                        if (!this._selectedRows || typeof (this._selectedRows[rowInfo.rowIndex]) !== "number") {
                            this._selectRow(rowInfo.rowIndex, rowInfo.dataIndex);
                        }
                    }

                    return this._handleEvent(e, this.onGutterClick, "gutterclick", {
                        gutterRow: rowInfo.gutterRow,
                        rowInfo: rowInfo,
                        rowIndex: rowInfo.rowIndex,
                        dataIndex: rowInfo.dataIndex
                    });
                }
            },

            onGutterClick: function (eventArgs) {
                if (this._options.gutter.contextMenu) {
                    this._showContextMenu(eventArgs && eventArgs.rowInfo);
                }
            },

            _onEnterKey: function (e) {
                if (this._handleEvent(e, this.onEnterKey, "enterkey") !== false) {
                    return this._onOpenRowDetail(e);
                }
                else {
                    return false;
                }
            },

            onEnterKey: function (eventArgs) {
                if (eventArgs.event.altKey) {
                    this._showContextMenu(eventArgs.rowInfo);
                    return false;
                }
            },

            _onDeleteKey: function (e) {
                return this._handleEvent(e, this.onDeleteKey, "deletekey");
            },

            onDeleteKey: function (eventArgs) {
            },

            _onOpenRowDetail: function (e, eventArgs) {
                var navigateOption, location, target;
                if (this._handleEvent(e, this.onOpenRowDetail, "openRowDetail", eventArgs) !== false) {
                    navigateOption = this._options.openRowDetail;
                    if (navigateOption) {
                        if ($.isFunction(navigateOption)) {
                            navigateOption = navigateOption.call(this, this._selectedIndex);
                        }

                        // only try to navigate if the function returned a location to navigate to.
                        if (navigateOption) {
                            if (typeof (navigateOption) !== "string") {
                                location = this.getColumnValue(this._selectedIndex, navigateOption.hrefIndex, -1);
                                target = navigateOption.target;
                            }
                            else {
                                location = navigateOption;
                            }

                            if (location) {
                                if (target) {
                                    TFSHOST.ActionManager.performAction(TFSHOST.CommonActions.ACTION_WINDOW_OPEN, {
                                        url: location,
                                        target: target,
                                        features: this._options.openRowDetail.targetFeatures
                                    });
                                }
                                else {
                                    TFSHOST.ActionManager.performAction(TFSHOST.CommonActions.ACTION_WINDOW_NAVIGATE, {
                                        url: location
                                    });
                                }
                            }
                        }

                        return false;
                    }
                }
                else {
                    return false;
                }
            },

            onOpenRowDetail: function (eventArgs) {
            },

            _onContextMenu: function (e, eventArgs) {
                this._handleEvent(e, this.onContextMenu, "contextMenu", eventArgs);

                return false;
            },

            onContextMenu: function (eventArgs) {
                this._showContextMenu(eventArgs && eventArgs.rowInfo);
            },

            _onBlur: function (e) {
                log(verbose, "(grid) focus out");

                this._active = false;

                this._updateSelectionStyles();
            },

            _onFocus: function (e) {
                log(verbose, "(grid) focus in");

                this._active = true;
                this._updateSelectionStyles();
                this._updateAriaAttribute();
            },

            _onKeyDown: function (e) {
                var bounds = { lo: -1, hi: -1 },
                    keyCode = TFS.UI.KeyCode,
                    canvas = this._canvas,
                    span,
                    rowsPerPage;

                log(verbose, "(grid) key down");

                if (this._copyInProgress) {
                    if (e.keyCode === keyCode.ESCAPE) {
                        if (this._cancelable) {
                            this._cancelable.cancel();
                        }
                    }

                    //TODO:cancel only our keys
                    // Cancelling this key
                    return false;
                }

                if (this._count > 0) {
                    bounds = { lo: 0, hi: this._expandedCount - 1 };
                }

                if (this._selectedIndex < 0) {
                    this._addSelection(bounds.lo);
                }

                switch (e.keyCode) {
                    case 65: // A
                        if (e.ctrlKey) {
                            this._selectAll();
                        }
                        else {
                            return;
                        }
                        break;
                    case 121: //F10 key
                        if (e.shiftKey) {
                            return this._onContextMenu();
                        }
                        else {
                            return;
                        }
                        break;
                    case keyCode.DOWN:
                        this._clearSelection();
                        if (e.ctrlKey) {
                            if (e.shiftKey) {
                                this._addSelectionRange(bounds.hi);
                            }
                            else {
                                this._addSelection(bounds.hi);
                            }
                        }
                        else {
                            if (e.shiftKey) {
                                this._addSelectionRange(Math.min(this._selectedIndex + 1, bounds.hi));
                            }
                            else {
                                this._addSelection(Math.min(this._selectedIndex + 1, bounds.hi));
                            }
                        }
                        break;

                    case keyCode.UP:
                        this._clearSelection();
                        if (e.ctrlKey) {
                            if (e.shiftKey) {
                                this._addSelectionRange(bounds.lo);
                            }
                            else {
                                this._addSelection(bounds.lo);
                            }
                        }
                        else {
                            if (e.shiftKey) {
                                this._addSelectionRange(Math.max(this._selectedIndex - 1, bounds.lo));
                            }
                            else {
                                this._addSelection(Math.max(this._selectedIndex - 1, bounds.lo));
                            }
                        }
                        break;

                    case keyCode.PAGE_DOWN:
                    case keyCode.PAGE_UP:
                        span = canvas[0].clientHeight;
                        rowsPerPage = Math.floor(span / this._rowHeight);

                        this._clearSelection();

                        if (e.keyCode === keyCode.PAGE_DOWN) {
                            if (e.shiftKey) {
                                this._addSelectionRange(Math.min(this._selectedIndex + rowsPerPage, bounds.hi));
                            }
                            else {
                                this._addSelection(Math.min(this._selectedIndex + rowsPerPage, bounds.hi));
                            }
                        }
                        else {
                            if (e.shiftKey) {
                                this._addSelectionRange(Math.max(this._selectedIndex - rowsPerPage, bounds.lo));
                            }
                            else {
                                this._addSelection(Math.max(this._selectedIndex - rowsPerPage, bounds.lo));
                            }
                        }
                        break;

                    case keyCode.RIGHT:
                        if (!this.tryToggle(true, e.shiftKey)) {
                            this._addSelection(Math.min(this._selectedIndex + 1, bounds.hi));
                        }
                        else {
                            this._clearSelection();
                            this._addSelection(this._selectedIndex);
                        }
                        break;

                    case keyCode.LEFT:
                        if (!this.tryToggle(false, e.shiftKey)) {
                            this._addSelection(Math.max(this._selectedIndex - 1, bounds.lo));
                        }
                        else {
                            this._clearSelection();
                            this._addSelection(this._selectedIndex);
                        }
                        break;

                    case keyCode.ENTER:
                        return this._onEnterKey(e);

                    case keyCode.DELETE:
                        return this._onDeleteKey(e);

                    case keyCode.HOME:
                        this._clearSelection();
                        if (e.shiftKey) {
                            this._addSelectionRange(bounds.lo);
                        }
                        else {
                            this._addSelection(bounds.lo);
                        }
                        break;

                    case keyCode.END:
                        this._clearSelection();
                        if (e.shiftKey) {
                            this._addSelectionRange(bounds.hi);
                        }
                        else {
                            this._addSelection(bounds.hi);
                        }
                        break;

                    case keyCode.ESCAPE:
                        this._tryFinishColumnMoving(true);
                        break;

                    default:
                        return;
                }

                this.getSelectedRowIntoView();
                this._fire("keydown", [e.keyCode, e.ctrlKey, e.shiftKey, e.altKey]);

                e.preventDefault();
            },

            _onKeyUp: function (e) {

            },

            _onToggle: function (rowInfo) {
                var state;

                if (this._expandStates) {
                    state = this._expandStates[rowInfo.dataIndex];

                    if (state !== 0) {
                        if (state > 0) {
                            this.collapseNode(rowInfo.dataIndex);
                        }
                        else if (state < 0) {
                            this.expandNode(rowInfo.dataIndex);
                        }

                        this._clearSelection();
                        this._addSelection(Math.min(rowInfo.rowIndex, this._expandedCount - 1), false, rowInfo.dataIndex);
                        this._layoutContentSpacer();
                        this._redraw();
                    }
                }
            },

            enableEvent: function (eventName) {
                /// <summary>Enables raising the custom event with the provided event name.</summary>
                /// <param name="eventName" type="String">Name of the event to enable.</param>

                Diag.assertParamIsStringNotEmpty(eventName, "eventName");

                var enabledEvents = this._options.enabledEvents;

                if (!enabledEvents) {
                    enabledEvents = {};
                    this._options.enabledEvents = enabledEvents;
                }

                enabledEvents[eventName] = true;
            },

            disableEvent: function (eventName) {
                /// <summary>Disables raising the custom event with the provided event name.</summary>
                /// <param name="eventName" type="String">Name of the event to disable.</param>

                Diag.assertParamIsStringNotEmpty(eventName, "eventName");

                var enabledEvents = this._options.enabledEvents;

                if (enabledEvents) {
                    delete enabledEvents[eventName];
                }
            },

            getExpandStates: function () {
                /// <summary>Gets the collection of expand states for the grid.</summary>
                return this._expandStates;
            },

            _copySelectedItems: function () {
                var grid = this,
                    options = {
                        cancellable: true,
                        message: Resources.CopyProgressPleaseWait,
                        cancelCallback: delegate(this, function () {
                            // put cancel operation call here.
                        })
                    },
                    longRunningOperation = new CommonControls.LongRunningOperation(grid._element, options);

                TFS.Diag.logTracePoint('TFS.UI.Controls.Grids._copySelectedItems.start');

                longRunningOperation.beginOperation(delegate(grid, function (cancelable) {
                    grid._cancelable = cancelable;
                    grid._beginCopySelection(function () {
                        longRunningOperation.endOperation();
                        TFS.Diag.logTracePoint('TFS.UI.Controls.Grids._copySelectedItems.complete');
                    });
                }));
            },

            beginFormatTable: function (operationCompleteCallback, errorCallback) {
                /// <summary>Generates a table of the selected items in the grid.</summary>
                /// <param name="operationCompleteCallback" type="Function">A callback function invoked when the
                /// current selection is available to the client for processing.</param>

                Diag.assertParamIsFunction(operationCompleteCallback, "operationCompleteCallback");

                this._beginEnsureSelectionIsAvailable(delegate(this, function () {
                    var grid = this,
                        formatter,
                        selectionText;

                    if (!(grid._cancelable && grid._cancelable.canceled)) {
                        formatter = new Data.TabDelimitedTableFormatter(grid);
                        selectionText = formatter.getTableFromSelectedItems();
                    }

                    if ($.isFunction(operationCompleteCallback)) {
                        operationCompleteCallback(selectionText);
                    }
                }, errorCallback));
            },

            _beginEnsureSelectionIsAvailable: function (itemsAvailableCallback, errorCallback) {
                /// <summary>Ensures that all data objects in the selection have been downloaded and are available to process.</summary>
                /// <remarks>The default implementation does nothing.  Grid implementors must implement this function.</remarks>

                Diag.assertParamIsFunction(itemsAvailableCallback);
                itemsAvailableCallback();
            },

            _beginCopySelection: function (operationCompleteCallback, errorCallback) {
                /// <summary>Copies the selection into the clipboard.</summary>

                if (this._copyInProgress) {
                    return;
                }

                // If there is no selection, don't bother trying to page anything down.
                if ((this.getSelectionCount().length <= 0) && $.isFunction(operationCompleteCallback)) {
                    operationCompleteCallback();
                }

                this._copyInProgress = true;
                this.beginFormatTable(delegate(this, function (table) {
                    if (!(this._cancelable && this._cancelable.canceled)) {
                        CommonControls.Clipboard.copyToClipboard(table);
                    }

                    this._copyInProgress = false;

                    if ($.isFunction(operationCompleteCallback)) {
                        operationCompleteCallback();
                    }
                }), errorCallback);
            }
        };
    }()));




    // WORKAROUND 798762 IE9: builds scroll bar for grid causes context menu icon to move to the middle when scrolled //
    $(function () {

        var mainContainer;

        // Applying this work around only for IE
        if ($.browser.msie) {
            // We should do this only for main-container for now because if we want
            // to walk up to the DOM tree for each grid, things get complicated if there
            // is more than 1 grid on the page
            mainContainer = $(document.body).children(".main-container")[0];

            function fix() {
                if (mainContainer) {
                    // Checking whether scrollbar exists or not.
                    // If scrollbar exists, setting container overflow to scroll to get grid working correctly,
                    // if not, setting container overflow back to auto to let it decide to show scrollbar or not
                    mainContainer.style.overflowX = mainContainer.clientWidth !== mainContainer.scrollWidth ? "scroll" : "auto";
                }
            }

            // Attaching window resize to fix the container overflow
            $(window).resize(function (e) {
                // jQuery UI dialog fires synthetic resize events
                if (e.target === window) {
                    fix();
                }
            });

            // Initial fixing (in case the initial state of the browser window is too small)
            fix();
        }
    });

    return {
        Grid: Grid,
    };
});

// SIG // Begin signature block
// SIG // MIIaswYJKoZIhvcNAQcCoIIapDCCGqACAQExCzAJBgUr
// SIG // DgMCGgUAMGcGCisGAQQBgjcCAQSgWTBXMDIGCisGAQQB
// SIG // gjcCAR4wJAIBAQQQEODJBs441BGiowAQS9NQkAIBAAIB
// SIG // AAIBAAIBAAIBADAhMAkGBSsOAwIaBQAEFD/X0TQ/XCZe
// SIG // iAyX4o7WyGT5INntoIIVeTCCBLowggOioAMCAQICCmEC
// SIG // kkoAAAAAACAwDQYJKoZIhvcNAQEFBQAwdzELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEhMB8GA1UEAxMYTWljcm9zb2Z0IFRp
// SIG // bWUtU3RhbXAgUENBMB4XDTEyMDEwOTIyMjU1OVoXDTEz
// SIG // MDQwOTIyMjU1OVowgbMxCzAJBgNVBAYTAlVTMRMwEQYD
// SIG // VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25k
// SIG // MR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24x
// SIG // DTALBgNVBAsTBE1PUFIxJzAlBgNVBAsTHm5DaXBoZXIg
// SIG // RFNFIEVTTjpCOEVDLTMwQTQtNzE0NDElMCMGA1UEAxMc
// SIG // TWljcm9zb2Z0IFRpbWUtU3RhbXAgU2VydmljZTCCASIw
// SIG // DQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAM1jw/ei
// SIG // tUfZ+TmUU6xrj6Z5OCH00W49FTgWwXMsmY/74Dxb4aJM
// SIG // i7Kri7TySse5k1DRJvWHU7B6dfNHDxcrZyxk62DnSozg
// SIG // i17EVmk3OioEXRcByL+pt9PJq6ORqIHjPy232OTEeAB5
// SIG // Oc/9x2TiIxJ4ngx2J0mPmqwOdOMGVVVJyO2hfHBFYX6y
// SIG // cRYe4cFBudLSMulSJPM2UATX3W88SdUL1HZA/GVlE36V
// SIG // UTrV/7iap1drSxXlN1gf3AANxa7q34FH+fBSrubPWqzg
// SIG // FEqmcZSA+v2wIzBg6YNgrA4kHv8R8uelVWKV7p9/ninW
// SIG // zUsKdoPwQwTfBkkg8lNaRLBRejkCAwEAAaOCAQkwggEF
// SIG // MB0GA1UdDgQWBBTNGaxhTZRnK/avlHVZ2/BYAIOhOjAf
// SIG // BgNVHSMEGDAWgBQjNPjZUkZwCu1A+3b7syuwwzWzDzBU
// SIG // BgNVHR8ETTBLMEmgR6BFhkNodHRwOi8vY3JsLm1pY3Jv
// SIG // c29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNyb3Nv
// SIG // ZnRUaW1lU3RhbXBQQ0EuY3JsMFgGCCsGAQUFBwEBBEww
// SIG // SjBIBggrBgEFBQcwAoY8aHR0cDovL3d3dy5taWNyb3Nv
// SIG // ZnQuY29tL3BraS9jZXJ0cy9NaWNyb3NvZnRUaW1lU3Rh
// SIG // bXBQQ0EuY3J0MBMGA1UdJQQMMAoGCCsGAQUFBwMIMA0G
// SIG // CSqGSIb3DQEBBQUAA4IBAQBRHNbfNh3cgLwCp8aZ3xbI
// SIG // kAZpFZoyufNkENKK82IpG3mPymCps13E5BYtNYxEm/H0
// SIG // XGGkQa6ai7pQ0Wp5arNijJ1NUVALqY7Uv6IQwEfVTnVS
// SIG // iR4/lmqPLkAUBnLuP3BZkl2F7YOZ+oKEnuQDASETqyfW
// SIG // zHFJ5dod/288CU7VjWboDMl/7jEUAjdfe2nsiT5FfyVE
// SIG // 5x8a1sUaw0rk4fGEmOdP+amYpxhG7IRs7KkDCv18elId
// SIG // nGukqA+YkqSSeFwreON9ssfZtnB931tzU7+q1GZQS/DJ
// SIG // O5WF5cFKZZ0lWFC7IFSReTobB1xqVyivMcef58Md7kf9
// SIG // J9d/z3TcZcU/MIIE7DCCA9SgAwIBAgITMwAAALARrwqL
// SIG // 0Duf3QABAAAAsDANBgkqhkiG9w0BAQUFADB5MQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQg
// SIG // Q29kZSBTaWduaW5nIFBDQTAeFw0xMzAxMjQyMjMzMzla
// SIG // Fw0xNDA0MjQyMjMzMzlaMIGDMQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMQ0wCwYDVQQLEwRNT1BSMR4wHAYDVQQDExVNaWNy
// SIG // b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEB
// SIG // AQUAA4IBDwAwggEKAoIBAQDor1yiIA34KHy8BXt/re7r
// SIG // dqwoUz8620B9s44z5lc/pVEVNFSlz7SLqT+oN+EtUO01
// SIG // Fk7vTXrbE3aIsCzwWVyp6+HXKXXkG4Unm/P4LZ5BNisL
// SIG // QPu+O7q5XHWTFlJLyjPFN7Dz636o9UEVXAhlHSE38Cy6
// SIG // IgsQsRCddyKFhHxPuRuQsPWj/ov0DJpOoPXJCiHiquMB
// SIG // Nkf9L4JqgQP1qTXclFed+0vUDoLbOI8S/uPWenSIZOFi
// SIG // xCUuKq6dGB8OHrbCryS0DlC83hyTXEmmebW22875cHso
// SIG // AYS4KinPv6kFBeHgD3FN/a1cI4Mp68fFSsjoJ4TTfsZD
// SIG // C5UABbFPZXHFAgMBAAGjggFgMIIBXDATBgNVHSUEDDAK
// SIG // BggrBgEFBQcDAzAdBgNVHQ4EFgQUWXGmWjNN2pgHgP+E
// SIG // Hr6H+XIyQfIwUQYDVR0RBEowSKRGMEQxDTALBgNVBAsT
// SIG // BE1PUFIxMzAxBgNVBAUTKjMxNTk1KzRmYWYwYjcxLWFk
// SIG // MzctNGFhMy1hNjcxLTc2YmMwNTIzNDRhZDAfBgNVHSME
// SIG // GDAWgBTLEejK0rQWWAHJNy4zFha5TJoKHzBWBgNVHR8E
// SIG // TzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NybC9wcm9kdWN0cy9NaWNDb2RTaWdQQ0Ff
// SIG // MDgtMzEtMjAxMC5jcmwwWgYIKwYBBQUHAQEETjBMMEoG
// SIG // CCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NlcnRzL01pY0NvZFNpZ1BDQV8wOC0zMS0y
// SIG // MDEwLmNydDANBgkqhkiG9w0BAQUFAAOCAQEAMdduKhJX
// SIG // M4HVncbr+TrURE0Inu5e32pbt3nPApy8dmiekKGcC8N/
// SIG // oozxTbqVOfsN4OGb9F0kDxuNiBU6fNutzrPJbLo5LEV9
// SIG // JBFUJjANDf9H6gMH5eRmXSx7nR2pEPocsHTyT2lrnqkk
// SIG // hNrtlqDfc6TvahqsS2Ke8XzAFH9IzU2yRPnwPJNtQtjo
// SIG // fOYXoJtoaAko+QKX7xEDumdSrcHps3Om0mPNSuI+5PNO
// SIG // /f+h4LsCEztdIN5VP6OukEAxOHUoXgSpRm3m9Xp5QL0f
// SIG // zehF1a7iXT71dcfmZmNgzNWahIeNJDD37zTQYx2xQmdK
// SIG // Dku/Og7vtpU6pzjkJZIIpohmgjCCBbwwggOkoAMCAQIC
// SIG // CmEzJhoAAAAAADEwDQYJKoZIhvcNAQEFBQAwXzETMBEG
// SIG // CgmSJomT8ixkARkWA2NvbTEZMBcGCgmSJomT8ixkARkW
// SIG // CW1pY3Jvc29mdDEtMCsGA1UEAxMkTWljcm9zb2Z0IFJv
// SIG // b3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5MB4XDTEwMDgz
// SIG // MTIyMTkzMloXDTIwMDgzMTIyMjkzMloweTELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEjMCEGA1UEAxMaTWljcm9zb2Z0IENv
// SIG // ZGUgU2lnbmluZyBQQ0EwggEiMA0GCSqGSIb3DQEBAQUA
// SIG // A4IBDwAwggEKAoIBAQCycllcGTBkvx2aYCAgQpl2U2w+
// SIG // G9ZvzMvx6mv+lxYQ4N86dIMaty+gMuz/3sJCTiPVcgDb
// SIG // NVcKicquIEn08GisTUuNpb15S3GbRwfa/SXfnXWIz6pz
// SIG // RH/XgdvzvfI2pMlcRdyvrT3gKGiXGqelcnNW8ReU5P01
// SIG // lHKg1nZfHndFg4U4FtBzWwW6Z1KNpbJpL9oZC/6SdCni
// SIG // di9U3RQwWfjSjWL9y8lfRjFQuScT5EAwz3IpECgixzdO
// SIG // PaAyPZDNoTgGhVxOVoIoKgUyt0vXT2Pn0i1i8UU956wI
// SIG // APZGoZ7RW4wmU+h6qkryRs83PDietHdcpReejcsRj1Y8
// SIG // wawJXwPTAgMBAAGjggFeMIIBWjAPBgNVHRMBAf8EBTAD
// SIG // AQH/MB0GA1UdDgQWBBTLEejK0rQWWAHJNy4zFha5TJoK
// SIG // HzALBgNVHQ8EBAMCAYYwEgYJKwYBBAGCNxUBBAUCAwEA
// SIG // ATAjBgkrBgEEAYI3FQIEFgQU/dExTtMmipXhmGA7qDFv
// SIG // pjy82C0wGQYJKwYBBAGCNxQCBAweCgBTAHUAYgBDAEEw
// SIG // HwYDVR0jBBgwFoAUDqyCYEBWJ5flJRP8KuEKU5VZ5KQw
// SIG // UAYDVR0fBEkwRzBFoEOgQYY/aHR0cDovL2NybC5taWNy
// SIG // b3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMvbWljcm9z
// SIG // b2Z0cm9vdGNlcnQuY3JsMFQGCCsGAQUFBwEBBEgwRjBE
// SIG // BggrBgEFBQcwAoY4aHR0cDovL3d3dy5taWNyb3NvZnQu
// SIG // Y29tL3BraS9jZXJ0cy9NaWNyb3NvZnRSb290Q2VydC5j
// SIG // cnQwDQYJKoZIhvcNAQEFBQADggIBAFk5Pn8mRq/rb0Cx
// SIG // MrVq6w4vbqhJ9+tfde1MOy3XQ60L/svpLTGjI8x8UJiA
// SIG // IV2sPS9MuqKoVpzjcLu4tPh5tUly9z7qQX/K4QwXacul
// SIG // nCAt+gtQxFbNLeNK0rxw56gNogOlVuC4iktX8pVCnPHz
// SIG // 7+7jhh80PLhWmvBTI4UqpIIck+KUBx3y4k74jKHK6BOl
// SIG // kU7IG9KPcpUqcW2bGvgc8FPWZ8wi/1wdzaKMvSeyeWNW
// SIG // RKJRzfnpo1hW3ZsCRUQvX/TartSCMm78pJUT5Otp56mi
// SIG // LL7IKxAOZY6Z2/Wi+hImCWU4lPF6H0q70eFW6NB4lhhc
// SIG // yTUWX92THUmOLb6tNEQc7hAVGgBd3TVbIc6YxwnuhQ6M
// SIG // T20OE049fClInHLR82zKwexwo1eSV32UjaAbSANa98+j
// SIG // Zwp0pTbtLS8XyOZyNxL0b7E8Z4L5UrKNMxZlHg6K3RDe
// SIG // ZPRvzkbU0xfpecQEtNP7LN8fip6sCvsTJ0Ct5PnhqX9G
// SIG // uwdgR2VgQE6wQuxO7bN2edgKNAltHIAxH+IOVN3lofvl
// SIG // RxCtZJj/UBYufL8FIXrilUEnacOTj5XJjdibIa4NXJzw
// SIG // oq6GaIMMai27dmsAHZat8hZ79haDJLmIz2qoRzEvmtzj
// SIG // cT3XAH5iR9HOiMm4GPoOco3Boz2vAkBq/2mbluIQqBC0
// SIG // N1AI1sM9MIIGBzCCA++gAwIBAgIKYRZoNAAAAAAAHDAN
// SIG // BgkqhkiG9w0BAQUFADBfMRMwEQYKCZImiZPyLGQBGRYD
// SIG // Y29tMRkwFwYKCZImiZPyLGQBGRYJbWljcm9zb2Z0MS0w
// SIG // KwYDVQQDEyRNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
// SIG // ZSBBdXRob3JpdHkwHhcNMDcwNDAzMTI1MzA5WhcNMjEw
// SIG // NDAzMTMwMzA5WjB3MQswCQYDVQQGEwJVUzETMBEGA1UE
// SIG // CBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEe
// SIG // MBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSEw
// SIG // HwYDVQQDExhNaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0Ew
// SIG // ggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCf
// SIG // oWyx39tIkip8ay4Z4b3i48WZUSNQrc7dGE4kD+7Rp9FM
// SIG // rXQwIBHrB9VUlRVJlBtCkq6YXDAm2gBr6Hu97IkHD/cO
// SIG // BJjwicwfyzMkh53y9GccLPx754gd6udOo6HBI1PKjfpF
// SIG // zwnQXq/QsEIEovmmbJNn1yjcRlOwhtDlKEYuJ6yGT1VS
// SIG // DOQDLPtqkJAwbofzWTCd+n7Wl7PoIZd++NIT8wi3U21S
// SIG // tEWQn0gASkdmEScpZqiX5NMGgUqi+YSnEUcUCYKfhO1V
// SIG // eP4Bmh1QCIUAEDBG7bfeI0a7xC1Un68eeEExd8yb3zuD
// SIG // k6FhArUdDbH895uyAc4iS1T/+QXDwiALAgMBAAGjggGr
// SIG // MIIBpzAPBgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBQj
// SIG // NPjZUkZwCu1A+3b7syuwwzWzDzALBgNVHQ8EBAMCAYYw
// SIG // EAYJKwYBBAGCNxUBBAMCAQAwgZgGA1UdIwSBkDCBjYAU
// SIG // DqyCYEBWJ5flJRP8KuEKU5VZ5KShY6RhMF8xEzARBgoJ
// SIG // kiaJk/IsZAEZFgNjb20xGTAXBgoJkiaJk/IsZAEZFglt
// SIG // aWNyb3NvZnQxLTArBgNVBAMTJE1pY3Jvc29mdCBSb290
// SIG // IENlcnRpZmljYXRlIEF1dGhvcml0eYIQea0WoUqgpa1M
// SIG // c1j0BxMuZTBQBgNVHR8ESTBHMEWgQ6BBhj9odHRwOi8v
// SIG // Y3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0
// SIG // cy9taWNyb3NvZnRyb290Y2VydC5jcmwwVAYIKwYBBQUH
// SIG // AQEESDBGMEQGCCsGAQUFBzAChjhodHRwOi8vd3d3Lm1p
// SIG // Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY3Jvc29mdFJv
// SIG // b3RDZXJ0LmNydDATBgNVHSUEDDAKBggrBgEFBQcDCDAN
// SIG // BgkqhkiG9w0BAQUFAAOCAgEAEJeKw1wDRDbd6bStd9vO
// SIG // eVFNAbEudHFbbQwTq86+e4+4LtQSooxtYrhXAstOIBNQ
// SIG // md16QOJXu69YmhzhHQGGrLt48ovQ7DsB7uK+jwoFyI1I
// SIG // 4vBTFd1Pq5Lk541q1YDB5pTyBi+FA+mRKiQicPv2/OR4
// SIG // mS4N9wficLwYTp2OawpylbihOZxnLcVRDupiXD8WmIsg
// SIG // P+IHGjL5zDFKdjE9K3ILyOpwPf+FChPfwgphjvDXuBfr
// SIG // Tot/xTUrXqO/67x9C0J71FNyIe4wyrt4ZVxbARcKFA7S
// SIG // 2hSY9Ty5ZlizLS/n+YWGzFFW6J1wlGysOUzU9nm/qhh6
// SIG // YinvopspNAZ3GmLJPR5tH4LwC8csu89Ds+X57H2146So
// SIG // dDW4TsVxIxImdgs8UoxxWkZDFLyzs7BNZ8ifQv+AeSGA
// SIG // nhUwZuhCEl4ayJ4iIdBD6Svpu/RIzCzU2DKATCYqSCRf
// SIG // WupW76bemZ3KOm+9gSd0BhHudiG/m4LBJ1S2sWo9iaF2
// SIG // YbRuoROmv6pH8BJv/YoybLL+31HIjCPJZr2dHYcSZAI9
// SIG // La9Zj7jkIeW1sMpjtHhUBdRBLlCslLCleKuzoJZ1GtmS
// SIG // hxN1Ii8yqAhuoFuMJb+g74TKIdbrHk/Jmu5J4PcBZW+J
// SIG // C33Iacjmbuqnl84xKf8OxVtc2E0bodj6L54/LlUWa8kT
// SIG // o/0xggSmMIIEogIBATCBkDB5MQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQgQ29kZSBTaWdu
// SIG // aW5nIFBDQQITMwAAALARrwqL0Duf3QABAAAAsDAJBgUr
// SIG // DgMCGgUAoIHIMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3
// SIG // AgEEMBwGCisGAQQBgjcCAQsxDjAMBgorBgEEAYI3AgEV
// SIG // MCMGCSqGSIb3DQEJBDEWBBS1k4XK1nnyBOlJY+/8jBeC
// SIG // 6RkeNTBoBgorBgEEAYI3AgEMMVowWKA+gDwAVABGAFMA
// SIG // LgBVAEkALgBDAG8AbgB0AHIAbwBsAHMALgBHAHIAaQBk
// SIG // AHMALgBkAGUAYgB1AGcALgBqAHOhFoAUaHR0cDovL21p
// SIG // Y3Jvc29mdC5jb20wDQYJKoZIhvcNAQEBBQAEggEANtZt
// SIG // ochK1f9F241ZW64aLa3mKDA9gbEnowszo5pruLX8VVm6
// SIG // CG/QGID8sh0lNYwsto7qxIiiONkKEllMFvTQ6c1eh2Ws
// SIG // lk7htZhJHIHRsQ4UUzx6WqKn0uVnsNCyZKuzOIOYYfZ9
// SIG // JesylYt64TeqGUzm7a/h2OLuDpW7cTQgGhjrhF9LQEOr
// SIG // vvtCQfQAo3NE4P5HsszGvzEoUoQlmB9rnZtz42ZWxciY
// SIG // vR5VjeMEh8abneHxd0IVzM2hlYkoisWEiKvYVLAx196B
// SIG // SvKyVL/f7wuIdRKACDKlqYr36n8jX8ddvbB8j+DuLrJC
// SIG // QzfizmoIxa60XkFPNoBRXRxT3YctY6GCAh8wggIbBgkq
// SIG // hkiG9w0BCQYxggIMMIICCAIBATCBhTB3MQswCQYDVQQG
// SIG // EwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UE
// SIG // BxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENv
// SIG // cnBvcmF0aW9uMSEwHwYDVQQDExhNaWNyb3NvZnQgVGlt
// SIG // ZS1TdGFtcCBQQ0ECCmECkkoAAAAAACAwCQYFKw4DAhoF
// SIG // AKBdMBgGCSqGSIb3DQEJAzELBgkqhkiG9w0BBwEwHAYJ
// SIG // KoZIhvcNAQkFMQ8XDTEzMDMxNTA2MzM1NVowIwYJKoZI
// SIG // hvcNAQkEMRYEFC/+b3fxDp50CrVc0yx+GuwioDQ/MA0G
// SIG // CSqGSIb3DQEBBQUABIIBAJy36XUPN7CruljWoFuQDhth
// SIG // cFC0GU/MxKy7/AGJvzMfffCJwokbbiSKLhGpLJaN9nFs
// SIG // 2LtMHEe1aPERR6wZY6LJ9QWNQ84lm+mpzi/jQizp1RT8
// SIG // l+leKJEDN/F7/nBMCgGxseel6dcKJLqDIZG/gHY0QjTm
// SIG // VjGg7an7toOvHFY7R33/dpVulu/Lc5DWfgsvwBrW80uo
// SIG // 7Yt0c4wgqs75gv4A58c8+2PP0N2q7doYY18t3lAch9/j
// SIG // yZRSHLruWGV6WbqgQ72Y417hPZoRTy3T3pYJePETrsNx
// SIG // 4v6E4fuRht78qLcIYg2c0MPqwYkOTCCVXAWbLP2zEbJP
// SIG // FYpNwnOWBMU=
// SIG // End signature block
